# Pre-Deployment Testing Guide

**Purpose:** Complete testing checklist before deploying to production server

**Date Created:** 2025-10-31
**Phase:** 8 Pre-Deployment Testing

---

## Overview

Before deploying to the production server, you must verify that the entire workflow functions correctly on your local Windows machine. This ensures no issues exist before server deployment.

---

## Prerequisites

### ✅ Verify Setup Complete

1. **Check Python environment:**
   ```bash
   py -3 --version
   # Should show Python 3.8+
   ```

2. **Verify virtual environment:**
   ```bash
   cd "c:\Apps\Email Reports"
   venv\Scripts\activate
   pip list
   # Should show all dependencies: google-api-python-client, pdfplumber, Jinja2, RapidFuzz, premailer, python-dotenv
   ```

3. **Check OAuth credentials exist:**
   ```bash
   dir credentials.json
   dir token.json
   # Both files should exist in root directory
   ```

4. **Verify .env configuration:**
   ```bash
   type .env
   # Check all required variables are set
   ```

5. **Check client database exists:**
   ```bash
   dir data\clients.csv
   # File should exist with client data
   ```

---

## Testing Checklist

### Test 1: OAuth Authentication ✅

**Purpose:** Verify Gmail API access works

```bash
py -3 tests/test_oauth.py
```

**Expected Result:**
- ✅ "Successfully authenticated with Gmail API"
- ✅ Token refreshes if needed
- ✅ No authentication errors

**If Failed:**
- Check credentials.json exists
- Verify token.json is valid
- Re-run OAuth setup if needed

---

### Test 2: Client Database ✅

**Purpose:** Verify client data loads correctly

```bash
py -3 tests/test_client_database.py
```

**Expected Result:**
- ✅ All clients loaded from CSV
- ✅ No missing required fields
- ✅ Business names properly formatted
- ✅ Email addresses valid

**If Failed:**
- Open data/clients.csv in Excel
- Check for missing data
- Verify CSV format is correct
- Ensure no special characters causing issues

---

### Test 3: PDF Extraction ✅

**Purpose:** Verify PDF parsing works

**Prerequisite:** Have at least 2-3 test PDFs in `data/pdfs/`

```bash
py -3 tests/test_pdf_extraction.py
```

**Expected Result:**
- ✅ PDFs successfully parsed
- ✅ Business names extracted
- ✅ KPI data extracted (all 6 metrics)
- ✅ No parsing errors

**If Failed:**
- Check PDF format hasn't changed
- Verify pdfplumber is installed
- Review PDF manually to see if readable
- Check logs/ for detailed error messages

---

### Test 4: KPI Extraction ✅

**Purpose:** Verify KPI data extraction accuracy

```bash
py -3 tests/test_kpi_extraction.py
```

**Expected Result:**
- ✅ All 6 KPIs extracted: Sessions, Conversions, Active Users, Engagement Rate, Bounce Rate, Avg Session Duration
- ✅ Values are correct when compared to PDF
- ✅ Formatting preserved (percentages, time format)

**If Failed:**
- Manually open a PDF and compare values
- Check if table structure changed in Looker Studio
- Review extraction logic if format changed

---

### Test 5: Email Generation ✅

**Purpose:** Verify email template renders correctly

```bash
py -3 tests/test_email_generation_integration.py
```

**Expected Result:**
- ✅ HTML email generated
- ✅ All placeholders filled (no [MISSING] values)
- ✅ KPI table formatted correctly
- ✅ CSS properly inlined (premailer working)
- ✅ Brand colors applied (#1A4D2E, #F6E96B)
- ✅ Signature section included

**If Failed:**
- Check templates/email_template.html exists
- Verify Jinja2 is installed
- Review generated HTML in data/email_previews/
- Check premailer is working

---

### Test 6: Gmail Integration ✅

**Purpose:** Verify Gmail API read/write access

```bash
py -3 tests/test_gmail_integration.py
```

**Expected Result:**
- ✅ Can read emails from Gmail
- ✅ Can create draft emails
- ✅ Draft appears in Gmail Drafts folder
- ✅ No API quota errors

**If Failed:**
- Check OAuth scopes include gmail.readonly, gmail.compose
- Verify Gmail API enabled in Google Cloud Console
- Check internet connection
- Review API quota limits

---

### Test 7: Approval Workflow ✅

**Purpose:** Verify approval tracking works

```bash
py -3 tests/test_approval_setup.py
```

**Expected Result:**
- ✅ approval_tracking.csv created
- ✅ Approval review HTML generated
- ✅ Status tracking works (Pending, Approved, Needs Revision)
- ✅ Can mark emails for approval

**If Failed:**
- Check data/ folder is writable
- Verify CSV format is correct
- Review approval_workflow.py for errors

---

### Test 8: Full Workflow (Dry Run) ✅

**Purpose:** Run complete end-to-end workflow with test data

**Important:** This uses real Gmail API but processes test PDFs only

```bash
py -3 main.py --process-pdfs
```

**What This Does:**
1. Loads test PDFs from data/pdfs/
2. Extracts business names and KPIs
3. Matches to client database
4. Generates HTML emails
5. Creates approval tracking
6. Shows summary report

**Expected Result:**
```
============================================================
WORKFLOW SUMMARY
============================================================
Total PDFs processed: 3
Emails generated: 3
Approved: 0
Pending: 3
Needs Revision: 0
============================================================
```

**Manual Verification Steps:**

1. **Check generated emails:**
   ```bash
   start data\email_previews\
   ```
   - Open each HTML file in browser
   - Verify all data looks correct
   - Check brand colors applied
   - Verify signature section present

2. **Check approval tracking:**
   ```bash
   start data\approval_tracking.csv
   ```
   - All clients listed
   - Status shows "Pending"
   - No extraction errors

3. **Review logs:**
   ```bash
   type logs\email_reports_2025-10-31.log
   ```
   - No ERROR messages
   - All INFO messages look correct
   - Processing completed successfully

**If Failed:**
- Check which step failed in logs
- Re-run specific test for that component
- Review error messages carefully
- Fix issues before proceeding

---

### Test 9: Draft Creation Test ✅

**Purpose:** Verify Gmail draft creation works

**Prerequisites:**
- Test 8 completed successfully
- Have at least 1 email approved in approval_tracking.csv

**Steps:**

1. **Approve one test email:**
   ```bash
   notepad data\approval_tracking.csv
   ```
   - Change Status from "Pending" to "Approved" for one client
   - Save file

2. **Create draft:**
   ```bash
   py -3 main.py --create-drafts
   ```

3. **Verify in Gmail:**
   - Open Gmail in browser
   - Go to Drafts folder
   - Find the test draft
   - Open it and verify:
     - ✅ Correct recipient email
     - ✅ Subject line correct
     - ✅ HTML email displays properly
     - ✅ PDF attached
     - ✅ Brand colors visible

4. **Delete test draft:**
   - Delete the test draft from Gmail
   - Don't send it (this is just a test)

**Expected Result:**
- ✅ Draft created successfully
- ✅ Draft appears in Gmail
- ✅ All content correct
- ✅ PDF attached properly

**If Failed:**
- Check Gmail API has gmail.compose scope
- Verify internet connection
- Check OAuth token is valid
- Review gmail_sender.py for errors

---

### Test 10: Cross-Platform Email Compatibility ✅

**Purpose:** Verify email renders correctly in different email clients

```bash
py -3 tools\verification\check_cross_platform.py
```

**Expected Result:**
- ✅ CSS properly inlined
- ✅ No external stylesheets
- ✅ Table-based layout used
- ✅ Brand colors in inline styles
- ✅ Compatible with Gmail, Outlook, Apple Mail

**Manual Test:**
1. Send one test email to yourself
2. Check on multiple devices/clients:
   - Gmail web
   - Gmail mobile app
   - Outlook (if available)
   - Apple Mail (if available)

**If Failed:**
- Review premailer implementation
- Check email_template.html structure
- Verify CSS inlining working
- Test with Litmus or Email on Acid (paid tools)

---

## Complete Pre-Deployment Checklist

Before proceeding to server deployment, verify all tests passed:

- [ ] Test 1: OAuth Authentication ✅
- [ ] Test 2: Client Database ✅
- [ ] Test 3: PDF Extraction ✅
- [ ] Test 4: KPI Extraction ✅
- [ ] Test 5: Email Generation ✅
- [ ] Test 6: Gmail Integration ✅
- [ ] Test 7: Approval Workflow ✅
- [ ] Test 8: Full Workflow (Dry Run) ✅
- [ ] Test 9: Draft Creation Test ✅
- [ ] Test 10: Cross-Platform Compatibility ✅

**Additional Manual Checks:**

- [ ] All client data in clients.csv is accurate and up-to-date
- [ ] .env file has correct agency information
- [ ] OAuth credentials (credentials.json, token.json) are valid
- [ ] Test PDFs available in data/pdfs/
- [ ] Virtual environment has all dependencies installed
- [ ] No ERROR messages in logs/
- [ ] System runs smoothly without crashes

---

## If All Tests Pass ✅

You are ready to proceed to:
1. **Server Deployment** - See [DEPLOYMENT_WORKFLOW.md](DEPLOYMENT_WORKFLOW.md)
2. **Production Testing** - Run first production cycle

---

## If Tests Fail ❌

### General Troubleshooting Steps

1. **Check logs first:**
   ```bash
   type logs\email_reports_2025-10-31.log
   ```

2. **Review error messages carefully:**
   - Look for specific file paths
   - Check for permission issues
   - Verify API credentials

3. **Common Issues:**

   **OAuth Errors:**
   - Re-run OAuth setup
   - Check credentials.json is valid
   - Verify Gmail API enabled

   **PDF Parsing Errors:**
   - Verify pdfplumber installed
   - Check PDF format hasn't changed
   - Test with different PDF

   **Database Errors:**
   - Check clients.csv format
   - Verify no missing required fields
   - Check for special characters

   **Email Generation Errors:**
   - Verify Jinja2 installed
   - Check premailer installed
   - Review template syntax

4. **Get Help:**
   - Review documentation in docs/user-guides/
   - Check TROUBLESHOOTING section in README.md
   - Review implementation docs in docs/implementation/

---

## Test Data Requirements

### Minimum Test Data Needed:

1. **Test PDFs:**
   - At least 2-3 Looker Studio PDFs in data/pdfs/
   - Mix of SEO and Google Ads reports (if applicable)
   - Recent format (within last 3 months)

2. **Client Database:**
   - At least 3-5 test clients in data/clients.csv
   - Matching business names to test PDFs
   - Valid email addresses (can use your own for testing)

3. **OAuth Credentials:**
   - Valid credentials.json
   - Valid token.json
   - Gmail API enabled

---

## Testing Timeline

**Estimated Time:** 2-3 hours

- Setup verification: 15 minutes
- Individual tests (1-7): 60 minutes
- Full workflow test (8): 30 minutes
- Draft creation test (9): 15 minutes
- Cross-platform test (10): 30 minutes
- Manual verification: 30 minutes

---

## Next Steps After Testing

Once all tests pass, proceed to:

📄 **[DEPLOYMENT_WORKFLOW.md](DEPLOYMENT_WORKFLOW.md)** - Complete workflow startup guide

---

**Testing Status:** ⏳ NOT STARTED
**Last Updated:** 2025-10-31
**Phase:** 8 - Pre-Deployment Testing
