# Email Reports Project Structure

## Root Directory

### Core Files
- **main.py** - Main entry point for the automation system
- **requirements.txt** - Python dependencies
- **README.md** - Project overview and getting started guide
- **CLAUDE.md** - Comprehensive project specification

### Configuration Files
- **.env** - Environment variables (NOT committed to git)
- **.env.example** - Example environment configuration
- **.env.sample** - Sample configuration template
- **.gitignore** - Git ignore rules
- **.mcp.json** - MCP configuration
- **credentials.json** - OAuth credentials (NOT committed to git)
- **token.json** - OAuth token (NOT committed to git)
- **token.pickle** - Legacy OAuth token (NOT committed to git)

## Directory Structure

### `/src` - Source Code
Main application modules:
- **orchestrator.py** - Main workflow orchestration
- **email_generator.py** - Email generation and templating
- **pdf_extractor.py** - PDF parsing and KPI extraction
- **client_database.py** - Client data management
- **gmail_reader.py** - Gmail API integration for PDF extraction
- **gmail_sender.py** - Gmail API integration for draft creation
- **drive_reader.py** - Google Drive integration
- **approval_workflow.py** - Approval process management
- **approval_tracker.py** - Approval tracking functionality
- **logger.py** - Centralized logging

### `/templates` - Email Templates
- **email_template.html** - Jinja2 HTML email template

### `/data` - Data Files
- **clients.csv** - Active client database
- **clients_backup_original.csv** - Original backup
- **clients_cleaned.csv** - Cleaned version
- **clients-sample.csv** - Sample client data
- **approval_tracking.csv** - Approval workflow tracking
- **last_generated_emails.json** - Last generation metadata
- **/pdfs/** - PDF storage directory
- **/archive/** - Archived PDFs
- **/email_previews/** - Generated email previews (gitignored)

### `/tests` - Test Suite
Formal test files:
- **test_client_database.py** - Client database tests
- **test_current_implementation.py** - Implementation tests
- **test_drive_download.py** - Drive integration tests
- **test_email_generation_integration.py** - Email generation tests
- **test_gmail_integration.py** - Gmail API tests
- **test_kpi_extraction.py** - PDF parsing tests
- **test_oauth.py** - OAuth authentication tests
- **test_pdf_extraction.py** - PDF extraction tests
- **test_approval_setup.py** - Approval workflow tests

### `/scripts` - Utility Scripts

#### `/scripts/utilities/`
- **setup_verify.py** - System setup verification
- **debug_google_ads.py** - Google Ads debugging

#### `/scripts/deployment/`
- **prepare_deployment.py** - Deployment preparation script

#### `/scripts/testing/`
- **run_approval_tests.py** - Approval workflow test runner
- **run_tests.py** - Main test runner

### `/tools` - Tools and Verification

#### `/tools/verification/`
- **check_cross_platform.py** - Cross-platform compatibility checker

### `/docs` - Documentation
User-facing and technical documentation:
- **QUICKSTART.md** - Quick start guide
- **USAGE.md** - Detailed usage instructions
- **WORKFLOW_GUIDE.md** - Workflow guide
- **OAUTH_SETUP_GUIDE.md** - OAuth setup instructions
- **SETUP_YOUR_ENV.md** - Environment setup guide

### `/config` - Configuration Files
Additional configuration files

### `/logs` - Log Files
Application logs (gitignored)

### `/output` - Generated Output
Generated files and reports (gitignored)

### `/research` - Research and Planning
Research documents and notes

### `/_Technical_Documentation` - Technical Docs
Detailed technical documentation and implementation notes

## Phase Documentation

### Phase Completion Documents
- **PHASE1_COMPLETION_STATUS.md** - Phase 1 summary
- **PHASE6_COMPLETION_SUMMARY.md** - Phase 6 summary
- **PHASE7_COMPLETION_SUMMARY.md** - Phase 7 summary
- **PHASE7_BRANDING_VERIFICATION.md** - Phase 7 branding verification
- **BRANDING_VERIFICATION.md** - Branding verification results
- **CROSS_PLATFORM_COMPATIBILITY.md** - Cross-platform testing results

### Implementation Documentation
- **DEVELOPMENT_PLAN.md** - Original development plan
- **DEPLOYMENT.md** - Deployment guide
- **EMAIL_GENERATOR_IMPLEMENTATION.md** - Email generator details
- **GMAIL_READER_IMPLEMENTATION.md** - Gmail reader implementation
- **IMPLEMENTATION_SUMMARY_PHASE3.md** - Phase 3 summary
- **APPROVAL_WORKFLOW_GUIDE.md** - Approval workflow guide
- **APPROVAL_WORKFLOW_IMPLEMENTATION.md** - Approval implementation
- **APPROVAL_WORKFLOW_QUICK_REFERENCE.md** - Quick reference
- **PDF_EXTRACTION_SUCCESS.md** - PDF extraction notes
- **PROJECT_STATUS.md** - Current project status
- **TASK_COMPLETION_SUMMARY.md** - Task summaries
- **NEXT_STEPS.md** - Next steps and planning

### Task Management
- **task_deps.md** - Task dependencies and tracking
- **START_HERE.md** - Project onboarding
- **example_emails.md** - Email examples

## Files Removed During Cleanup

### Temporary Test Files (Deleted)
- ~~test_branding.py~~ - Temporary branding test
- ~~test_signature_fix.py~~ - Temporary signature test
- ~~test_jinja_render.py~~ - Temporary Jinja test
- ~~test_premailer.py~~ - Temporary premailer test
- ~~nul~~ - Error file
- ~~data/test_jinja_output.html~~ - Test output
- ~~data/test_premailer_new.html~~ - Test output
- ~~data/test_premailer_old.html~~ - Test output
- ~~pdf_text_seo.txt~~ - Old PDF test
- ~~pdf_extraction_results.txt~~ - Old PDF test

## Git Ignore Rules

The following are excluded from version control:
- OAuth credentials (credentials.json, token.*)
- Environment files (.env)
- Data files (CSVs, PDFs)
- Generated HTML previews
- Python cache (__pycache__, *.pyc)
- Virtual environments (venv/)
- Log files
- Output files
- Temporary files
- IDE settings (.claude/, .vscode/, .idea/)

## Quick Reference

### Running the System
```bash
# Full workflow
py -3 main.py --full

# Process existing PDFs
py -3 main.py --process-pdfs

# Create drafts only
py -3 main.py --create-drafts
```

### Running Tests
```bash
# Run all tests
py -3 scripts/testing/run_tests.py

# Verify setup
py -3 scripts/utilities/setup_verify.py

# Check cross-platform compatibility
py -3 tools/verification/check_cross_platform.py
```

### Deployment
```bash
# Prepare for deployment
py -3 scripts/deployment/prepare_deployment.py
```

## Maintenance Notes

- Keep `/data/clients.csv` updated with current client information
- Review `/logs` directory periodically for errors
- Generated email previews in `/data/email_previews/` are for review only
- Archive old PDFs from `/data/pdfs/` to `/data/archive/` periodically
- Backup OAuth credentials regularly (stored outside git)

## Version Control

This project uses Git with the following key branches:
- **main** - Production-ready code
- Development follows feature branch workflow

Last updated: 2025-10-31
