# Email Reports Automation - Usage Guide

Comprehensive step-by-step guide for using the Email Reports Automation System.

## Table of Contents

1. [First-Time Setup](#first-time-setup)
2. [Monthly Workflow](#monthly-workflow)
3. [Command Reference](#command-reference)
4. [Approval Workflow](#approval-workflow)
5. [Email Customization](#email-customization)
6. [Database Management](#database-management)
7. [Advanced Usage](#advanced-usage)

---

## First-Time Setup

### Prerequisites

- Python 3.8+ installed on Windows
- Google Workspace Gmail account
- Google Cloud Console access

### Step 1: Install Python Dependencies

```bash
cd "c:\Apps\Email Reports"
venv\Scripts\activate
pip install -r requirements.txt
```

### Step 2: Configure Google Cloud Project

#### 2a. Create Project

1. Visit [Google Cloud Console](https://console.cloud.google.com/)
2. Click "Select a project" → "New Project"
3. Name: "Email Reports Automation"
4. Click "Create"

#### 2b. Enable APIs

1. Navigate to "APIs & Services" → "Library"
2. Enable "Gmail API":
   - Search "Gmail API"
   - Click "Enable"
3. Enable "Google Sheets API":
   - Search "Google Sheets API"
   - Click "Enable"
4. Enable "Google Drive API" (optional, for Drive extraction):
   - Search "Google Drive API"
   - Click "Enable"

#### 2c. Configure OAuth Consent Screen

1. Go to "APIs & Services" → "OAuth consent screen"
2. Choose "Internal" (for Google Workspace) or "External"
3. Fill in required fields:
   - App name: Email Reports Automation
   - User support email: your email
   - Developer contact: your email
4. Add scopes:
   - `https://www.googleapis.com/auth/gmail.readonly`
   - `https://www.googleapis.com/auth/gmail.compose`
   - `https://www.googleapis.com/auth/gmail.send`
   - `https://www.googleapis.com/auth/spreadsheets`
5. Save and continue

#### 2d. Create OAuth Credentials

1. Go to "APIs & Services" → "Credentials"
2. Click "Create Credentials" → "OAuth client ID"
3. Application type: "Desktop app"
4. Name: "Email Reports Desktop Client"
5. Click "Create"
6. Download JSON file
7. Save as `credentials.json` in `c:\Apps\Email Reports\`

### Step 3: First-Time Authorization

Run the OAuth authorization flow:

```bash
cd "c:\Apps\Email Reports"
venv\Scripts\python test_oauth.py
```

**What happens:**
1. Browser opens automatically
2. Sign in with your Google account
3. Grant permissions to the app
4. See "Authentication successful!" message
5. `token.json` file is created (stores your authorization)

**Important:** Keep `token.json` secure. It grants access to your Gmail account.

### Step 4: Configure Environment Variables

Copy the example environment file:

```bash
copy .env.example .env
```

Edit `.env` file with your information:

```ini
# Gmail Configuration
GMAIL_SENDER_EMAIL=youragency@gmail.com
LOOKER_STUDIO_SENDER=noreply-lookerstudio@google.com

# Paths (Windows format)
PROJECT_ROOT=c:/Apps/Email Reports
DATA_DIR=c:/Apps/Email Reports/data
CLIENT_DATABASE_PATH=c:/Apps/Email Reports/data/clients.csv

# Agency Branding
AGENCY_NAME=Your Digital Marketing Agency
AGENCY_EMAIL=contact@youragency.com
AGENCY_PHONE=(555) 123-4567
AGENCY_WEBSITE=https://www.youragency.com

# Email Content
STANDARD_SEO_PARAGRAPH=Your keyword rankings continue to improve across target search terms. We're monitoring performance closely and will continue optimizing your content strategy to maintain upward momentum.

STANDARD_SEM_PARAGRAPH=Your Google Ads campaigns continue to drive quality traffic and conversions. We're actively monitoring performance and making bid adjustments to maximize your ROI.

CLOSING_PARAGRAPH=Please review the attached PDF for your complete monthly report. If you have any questions or would like to discuss these results in more detail, don't hesitate to reach out.
```

### Step 5: Populate Client Database

Open `data\clients.csv` and add your 30 clients:

```csv
Client-Name,Contact-Name,Contact-Email,Service-Type,SEO-Introduction,Google-Ads-Introduction
ABC Corporation,John Smith,john@abc.com,SEO,Great work on the content updates last month.,
XYZ Services,Jane Doe,jane@xyz.com,SEM,,Your new ad copy is performing well.
Tech Solutions,Mike Johnson,mike@tech.com,SEO,The technical SEO improvements are paying off.,
```

**Important:** Business names in CSV must match (or closely match) business names in Looker Studio PDFs.

---

## Monthly Workflow

### Overview

Run this workflow at the start of each month when Looker Studio PDFs arrive.

### Step-by-Step Process

#### Step 1: Extract PDFs from Gmail

```bash
cd "c:\Apps\Email Reports"
venv\Scripts\python main.py --extract-from-gmail
```

**What happens:**
- Connects to Gmail API
- Searches for emails from Looker Studio
- Downloads all PDF attachments to `data/pdfs/`
- Marks emails as processed (applies Gmail label)

**Expected output:**
```
[INFO] Connected to Gmail API
[INFO] Found 30 emails from noreply-lookerstudio@google.com
[INFO] Extracting PDF: ABC Corporation - January 2025 SEO Report.pdf
[INFO] Extracting PDF: XYZ Services - January 2025 Google Ads Report.pdf
...
[INFO] Extracted 30 PDFs successfully
```

**Alternative:** Extract from Google Drive

```bash
venv\Scripts\python main.py --extract-from-drive
```

#### Step 2: Process PDFs and Generate Emails

```bash
venv\Scripts\python main.py --process-pdfs
```

**What happens:**
- Reads each PDF from `data/pdfs/`
- Extracts business name, date, KPIs
- Matches business name to client database
- Generates personalized HTML email
- Creates approval tracking Google Sheet
- Saves email previews to `data/email_previews/`

**Expected output:**
```
[INFO] Processing 30 PDFs
[INFO] Extracting data from: ABC Corporation - January 2025 SEO Report.pdf
[INFO] Business name: ABC Corporation
[INFO] Report type: SEO
[INFO] Matched client: ABC Corporation (John Smith)
[INFO] Generated email for ABC Corporation
...
[INFO] Created approval tracking sheet: https://docs.google.com/spreadsheets/d/...
[INFO] Processing complete: 28 emails generated, 2 flagged for review
```

#### Step 3: Review Emails in Google Sheets

1. Open the approval tracking Google Sheet (URL in console output)
2. Review each row:
   - **Client Name**: Business from PDF
   - **Contact Name**: Client contact from database
   - **Report Type**: SEO or Google Ads
   - **Status**: Pending/Approved/Needs Revision
   - **Extraction Errors**: Any issues found
   - **Notes**: Your review notes

3. For each email:
   - Click "Preview Email" link to see HTML email
   - Review business name, KPIs, personalized text
   - Change Status dropdown:
     - **Approved**: Email is correct, ready to send
     - **Needs Revision**: Found issues, requires manual fix
     - **Pending**: Not yet reviewed

4. Add notes for emails needing revision

5. Save changes (Google Sheets auto-saves)

#### Step 4: Create Gmail Drafts

After approving emails in Google Sheet:

```bash
venv\Scripts\python main.py --create-drafts
```

**What happens:**
- Reads approval status from Google Sheet
- Creates Gmail draft for each "Approved" email
- Attaches original PDF to draft
- Logs creation status

**Expected output:**
```
[INFO] Reading approval status from Google Sheets
[INFO] Found 28 approved emails
[INFO] Creating draft for ABC Corporation
[INFO] Creating draft for XYZ Services
...
[INFO] Created 28 Gmail drafts successfully
```

#### Step 5: Send Drafts from Gmail

1. Open Gmail in browser
2. Navigate to "Drafts" folder
3. You'll see 28 draft emails (one per approved client)

**For each draft:**
1. Open draft
2. Review email one final time
3. Verify:
   - Recipient email is correct
   - Subject line is correct
   - KPI data looks reasonable
   - PDF is attached
4. Click "Send"

**Recommended sending pace:**
- Send 10 emails
- Wait 15 minutes
- Send next 10 emails
- Wait 15 minutes
- Send final 8-10 emails

**Why space out sends?**
- Avoids Gmail spam filters
- Better deliverability
- Mimics human sending pattern

**Estimated time:** 30-45 minutes to send all 30 emails

---

## Command Reference

### Main Commands

```bash
# Full workflow (extract → process → create approval sheet)
python main.py --full

# Extract PDFs from Gmail
python main.py --extract-from-gmail

# Extract PDFs from Google Drive
python main.py --extract-from-drive

# Process PDFs and generate emails
python main.py --process-pdfs

# Create Gmail drafts (after approval)
python main.py --create-drafts

# Show help
python main.py --help
```

### Test Commands

```bash
# Test Gmail OAuth authentication
python test_oauth.py

# Test PDF extraction with sample PDF
python test_pdf_extraction.py data/pdfs/sample.pdf

# Test client database loading
python test_client_database.py

# Run all unit tests
python run_tests.py
```

### Troubleshooting Commands

```bash
# View recent logs
type logs\email_reports_2025-01-05.log

# List PDFs in storage
dir data\pdfs\*.pdf

# Verify client database
python -c "from src.client_database import ClientDatabase; db = ClientDatabase('data/clients.csv'); print(f'Loaded {len(db.clients)} clients')"
```

---

## Approval Workflow

### Google Sheets Approval Tracker

The approval tracker is a Google Sheet created automatically during processing.

#### Sheet Columns

| Column | Description | Values |
|--------|-------------|--------|
| Client Name | Business name from PDF | ABC Corporation |
| Contact Name | Contact from database | John Smith |
| Contact Email | Recipient email | john@abc.com |
| Report Type | SEO or Google Ads | SEO |
| Report Date | Month/Year | January 2025 |
| Status | Approval status | Pending / Approved / Needs Revision |
| Extraction Errors | Issues found | "Missing Bounce Rate" or blank |
| Preview Email | Link to HTML preview | Click to view |
| Notes | Your review notes | "Check KPI data" |
| Generated Date | When email was created | 2025-01-05 09:15:00 |

#### Status Values

- **Pending** (default): Not yet reviewed
- **Approved**: Ready to create Gmail draft
- **Needs Revision**: Issues found, requires manual fix

#### Review Checklist

For each email, verify:

- [ ] Business name matches client
- [ ] Contact name and email are correct
- [ ] Report type matches service (SEO client gets SEO report)
- [ ] Report date is current month
- [ ] All KPIs extracted successfully (no "N/A" unless expected)
- [ ] Personalized introduction text is appropriate
- [ ] No extraction errors listed

#### Handling "Needs Revision"

If you mark an email as "Needs Revision":

1. Add detailed notes in "Notes" column
2. Check the extraction error message
3. Common issues:
   - **Business name mismatch**: Update `clients.csv` to match PDF exactly
   - **Missing KPIs**: Manually review PDF, may need to extract manually
   - **Wrong service type**: Check PDF type vs. client database service type
4. Fix the underlying issue
5. Re-run `python main.py --process-pdfs` to regenerate
6. Review updated Google Sheet

---

## Email Customization

### Customizing Email Template

Template location: `templates/email_template.html`

**WARNING:** Be careful editing HTML email templates. Email clients (Gmail, Outlook) have strict rendering requirements.

#### Safe Customizations

1. **Update agency branding:**
   - Edit `.env` file (AGENCY_NAME, AGENCY_EMAIL, etc.)
   - No HTML editing required

2. **Change standard paragraphs:**
   - Edit `.env` file (STANDARD_SEO_PARAGRAPH, STANDARD_SEM_PARAGRAPH)
   - Plain text only (no HTML tags)

3. **Update closing text:**
   - Edit `.env` file (CLOSING_PARAGRAPH)

#### Advanced Customizations (Requires HTML Knowledge)

To modify email structure:

1. Open `templates/email_template.html`
2. Use Jinja2 syntax for dynamic content:
   ```html
   <p>Hi {{ first_name }},</p>
   <p>{{ personalized_text }}</p>
   ```
3. **Critical:** Use inline CSS only (no `<style>` tags)
4. **Critical:** Use table-based layouts (not divs/flexbox)
5. Test across email clients (Gmail, Outlook, Apple Mail)

#### Available Template Variables

- `{{ first_name }}` - Client's first name
- `{{ business_name }}` - Company name
- `{{ report_month }}` - Report month (e.g., "January")
- `{{ report_type }}` - "SEO" or "Google Ads"
- `{{ personalized_text }}` - Custom intro from database
- `{{ standard_paragraph }}` - Standard SEO/SEM paragraph
- `{{ kpis }}` - Dictionary of KPI data
- `{{ agency_name }}`, `{{ agency_email }}`, etc. - Agency branding

### Customizing Per-Client Text

Each client can have unique introductory text in `data/clients.csv`:

```csv
Client-Name,Contact-Name,Contact-Email,Service-Type,SEO-Introduction,Google-Ads-Introduction
ABC Corp,John,john@abc.com,SEO,Great work on the new landing pages.,
XYZ Inc,Jane,jane@xyz.com,SEM,,Your ad copy refresh is converting well.
```

- **SEO clients:** Use `SEO-Introduction` column
- **SEM clients:** Use `Google-Ads-Introduction` column
- Leave other column blank
- Keep text 1-2 sentences, conversational tone

---

## Database Management

### Adding New Clients

1. Open `data\clients.csv` in Excel or Google Sheets
2. Add new row with:
   - Client-Name: **Must match Looker Studio PDF business name exactly**
   - Contact-Name: Client's first name (or "First & Last" for couples)
   - Contact-Email: Where to send report
   - Service-Type: "SEO" or "SEM"
   - SEO-Introduction or Google-Ads-Introduction: Custom 1-2 sentence intro
3. Save as CSV
4. Test match:
   ```bash
   python -c "from src.client_database import ClientDatabase; db = ClientDatabase('data/clients.csv'); print(db.find_client('Your New Client Name'))"
   ```

### Updating Client Information

To change email address, contact name, or personalized text:

1. Open `data\clients.csv`
2. Find client row
3. Edit fields
4. Save file

**No need to re-run anything** - changes take effect on next monthly run.

### Deactivating Clients

If a client should no longer receive reports:

1. Open `data\clients.csv`
2. Find client row
3. Delete entire row **OR** add column "Active" and set to FALSE
4. Save file

### Backing Up Database

**Recommended:** Back up `data\clients.csv` weekly.

Methods:
- Copy to Google Drive
- Email to yourself
- Save to external USB drive
- Commit to private Git repository

**Restoration:**
```bash
# If you lose clients.csv, restore from backup
copy "c:\Backups\clients_backup_2025-01-01.csv" "c:\Apps\Email Reports\data\clients.csv"
```

---

## Advanced Usage

### Processing Subset of PDFs

To process only certain PDFs (testing or re-processing):

1. Move PDFs you want to process to `data/pdfs/`
2. Move other PDFs to `data/archive/`
3. Run `python main.py --process-pdfs`
4. Only PDFs in `data/pdfs/` will be processed

### Re-generating Emails After Fixes

If you need to regenerate emails after fixing database issues:

1. Fix issue in `data/clients.csv`
2. Delete old email previews: `del data\email_previews\*`
3. Re-run: `python main.py --process-pdfs`
4. New approval sheet will be created with updated data

### Manual Email Generation

To generate a single email manually (for testing):

```python
from src.pdf_extractor import PDFExtractor
from src.client_database import ClientDatabase
from src.email_generator import EmailGenerator

# Extract PDF data
extractor = PDFExtractor()
pdf_data = extractor.extract_report_data('data/pdfs/test.pdf')

# Find client
db = ClientDatabase('data/clients.csv')
client = db.find_client(pdf_data['business_name'])

# Generate email
generator = EmailGenerator('templates/email_template.html')
email_html, email_text = generator.generate_email(
    client_data=client,
    pdf_data=pdf_data
)

# Save to file
with open('test_email.html', 'w', encoding='utf-8') as f:
    f.write(email_html)

print("Email generated: test_email.html")
```

### Adjusting Fuzzy Matching Threshold

If business names aren't matching correctly:

Edit `.env`:
```ini
FUZZY_MATCH_THRESHOLD=85  # Default: 85 (range: 0-100)
```

- **Higher (90-95)**: Requires very close match, fewer false positives
- **Lower (75-80)**: Allows more variation, more false positives

### Viewing Detailed Logs

Logs are stored in `logs/` with daily filenames.

```bash
# View today's log
type logs\email_reports_2025-01-05.log

# View only errors
findstr /C:"[ERROR]" logs\email_reports_2025-01-05.log

# View specific module (e.g., PDF extraction)
findstr /C:"[PDFExtractor]" logs\email_reports_2025-01-05.log
```

---

## Tips & Best Practices

### Monthly Workflow Tips

1. **Wait for all PDFs**: Don't start processing until all 30 PDFs have arrived in Gmail
2. **Review before approving**: Spot-check at least 5-10 emails before batch-approving
3. **Check error messages**: Review any "Needs Revision" emails carefully
4. **Test first send**: Send first email to yourself to verify formatting

### Database Maintenance Tips

1. **Weekly backup**: Back up `clients.csv` every Monday
2. **Quarterly review**: Review personalized text every 3 months, keep it fresh
3. **Document changes**: Keep notes of database changes (who added, why)

### Email Sending Tips

1. **Space out sends**: Don't send all 30 at once
2. **Send during business hours**: 9 AM - 5 PM local time preferred
3. **Monitor bounce-backs**: Check for delivery failures
4. **Client feedback**: Note if clients report spam issues

### Troubleshooting Tips

1. **Check logs first**: Most errors are explained in logs
2. **Test with one PDF**: Process single PDF to isolate issues
3. **Verify credentials**: Re-run `test_oauth.py` if Gmail fails
4. **Keep token.json**: Don't delete this file accidentally

---

## Support

For additional help:

1. Review [README.md](README.md) for overview
2. Check [DEPLOYMENT.md](DEPLOYMENT.md) for server setup
3. View logs in `logs/` directory
4. Contact system administrator

---

**Email Reports Automation System - Usage Guide**
*Version 1.0 - January 2025*
