#!/usr/bin/env python3
"""
Deployment Preparation Script

This script prepares the Email Reports system for deployment to Linux server.
It creates a deployment package with all necessary files and generates deployment instructions.
"""

import os
import shutil
from pathlib import Path
from datetime import datetime


class DeploymentPreparation:
    def __init__(self, project_root: Path):
        self.project_root = project_root
        self.deploy_dir = project_root / "deploy"
        self.timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")

    def prepare_deployment_package(self):
        """Create deployment package ready for server upload."""
        print("="*80)
        print("EMAIL REPORTS - DEPLOYMENT PREPARATION")
        print("="*80)
        print(f"Project Root: {self.project_root}")
        print(f"Deploy Directory: {self.deploy_dir}")
        print(f"Timestamp: {self.timestamp}")
        print("="*80)

        # Step 1: Clean existing deploy directory
        self._clean_deploy_directory()

        # Step 2: Create deploy structure
        self._create_deploy_structure()

        # Step 3: Copy source files
        self._copy_source_files()

        # Step 4: Copy templates
        self._copy_templates()

        # Step 5: Copy data files (excluding PDFs)
        self._copy_data_files()

        # Step 6: Copy configuration files
        self._copy_config_files()

        # Step 7: Create Linux .env template
        self._create_linux_env_template()

        # Step 8: Create deployment checklist
        self._create_deployment_checklist()

        # Step 9: Create verification script
        self._create_verification_script()

        # Step 10: Generate deployment summary
        self._generate_deployment_summary()

    def _clean_deploy_directory(self):
        """Remove existing deploy directory if it exists."""
        if self.deploy_dir.exists():
            print(f"\n🗑️  Removing existing deploy directory...")
            shutil.rmtree(self.deploy_dir)
            print("   ✓ Cleaned")

    def _create_deploy_structure(self):
        """Create deployment directory structure."""
        print(f"\n📁 Creating deployment directory structure...")

        directories = [
            "src",
            "templates",
            "data",
            "logs",
            "tests",
            "data/pdfs",
            "data/email_previews",
            "data/archive",
        ]

        for dir_name in directories:
            dir_path = self.deploy_dir / dir_name
            dir_path.mkdir(parents=True, exist_ok=True)
            print(f"   ✓ Created: {dir_name}/")

    def _copy_source_files(self):
        """Copy Python source files."""
        print(f"\n📄 Copying source files...")

        src_dir = self.project_root / "src"
        dest_dir = self.deploy_dir / "src"

        for py_file in src_dir.glob("*.py"):
            shutil.copy2(py_file, dest_dir / py_file.name)
            print(f"   ✓ {py_file.name}")

    def _copy_templates(self):
        """Copy email templates."""
        print(f"\n📧 Copying email templates...")

        template_dir = self.project_root / "templates"
        dest_dir = self.deploy_dir / "templates"

        if template_dir.exists():
            for template_file in template_dir.glob("*"):
                if template_file.is_file():
                    shutil.copy2(template_file, dest_dir / template_file.name)
                    print(f"   ✓ {template_file.name}")

    def _copy_data_files(self):
        """Copy data files (excluding large PDFs)."""
        print(f"\n💾 Copying data files...")

        # Copy clients.csv if it exists
        clients_csv = self.project_root / "data" / "clients.csv"
        if clients_csv.exists():
            shutil.copy2(clients_csv, self.deploy_dir / "data" / "clients.csv")
            print(f"   ✓ clients.csv")
        else:
            print(f"   ⚠️  clients.csv not found (will need to upload separately)")

        # Note: We intentionally skip PDF files (too large, not needed on server initially)
        print(f"   ℹ️  Skipping PDF files (will be generated on server)")

    def _copy_config_files(self):
        """Copy configuration and requirements files."""
        print(f"\n⚙️  Copying configuration files...")

        files_to_copy = [
            "main.py",
            "requirements.txt",
            "README.md",
            "USAGE.md",
            "DEPLOYMENT.md",
            "CROSS_PLATFORM_COMPATIBILITY.md",
            ".env.example",
        ]

        for filename in files_to_copy:
            src_file = self.project_root / filename
            if src_file.exists():
                shutil.copy2(src_file, self.deploy_dir / filename)
                print(f"   ✓ {filename}")
            else:
                print(f"   ⚠️  {filename} not found")

        # Note: We do NOT copy .env, credentials.json, or token.json
        # These must be uploaded separately with updated paths
        print(f"\n   ℹ️  IMPORTANT: You must manually upload:")
        print(f"      - .env (update paths for Linux)")
        print(f"      - credentials.json")
        print(f"      - token.json")

    def _create_linux_env_template(self):
        """Create .env template with Linux paths."""
        print(f"\n🐧 Creating Linux .env template...")

        linux_env_content = """# Gmail Configuration
GMAIL_SENDER_EMAIL=your-agency@gmail.com
LOOKER_STUDIO_SENDER=looker-studio@google.com

# Google Drive Configuration (Shared Drive folder IDs)
# SEO Reports folder: Shared drives\\DWS\\SEO\\Reports\\2025
DRIVE_SEO_FOLDER_ID=1BpQKQcZ9YO0lJVKtywL0berbt-k9cjuc
# Google Ads Reports folder: Shared drives\\DWS\\Google Ads\\Reports\\2025
DRIVE_SEM_FOLDER_ID=1AoEF52KRdQFblgcKg2mfLJmhCgSGIGKv

# File Paths (LINUX PATHS - update 'username' to your cPanel username)
CLIENT_DATABASE_PATH=/home/username/email_reports/data/clients.csv
PDF_STORAGE_PATH=/home/username/email_reports/data/pdfs/
TEMPLATE_PATH=/home/username/email_reports/templates/email_template.html
APPROVAL_TRACKING_PATH=/home/username/email_reports/data/approval_tracking.csv

# Processing Configuration
FUZZY_MATCH_THRESHOLD=85
LOG_LEVEL=INFO
MAX_PDFS_PER_RUN=50

# Email Template Text
AGENCY_NAME=Your Agency Name
AGENCY_EMAIL=contact@youragency.com
AGENCY_PHONE=(555) 123-4567
AGENCY_WEBSITE=https://www.youragency.com
STANDARD_SEO_PARAGRAPH=Your keyword rankings continue to improve across target search terms. We're monitoring performance closely and will continue optimizing your content strategy to maintain upward momentum.
STANDARD_SEM_PARAGRAPH=Your Google Ads campaigns continue to drive quality traffic and conversions. We're actively monitoring performance and making bid adjustments to maximize your ROI.
STANDARD_CLOSING_PARAGRAPH=Please review the attached PDF for your complete monthly report. If you have any questions or would like to discuss these results in more detail, don't hesitate to reach out.
"""

        env_linux_file = self.deploy_dir / ".env.linux"
        with open(env_linux_file, 'w', encoding='utf-8') as f:
            f.write(linux_env_content)

        print(f"   ✓ Created .env.linux template")
        print(f"   ℹ️  Rename to .env on server and update 'username' placeholder")

    def _create_deployment_checklist(self):
        """Create deployment checklist."""
        print(f"\n✅ Creating deployment checklist...")

        checklist_content = """# DEPLOYMENT CHECKLIST

Email Reports Automation - Server Deployment
Date: {timestamp}

## Pre-Deployment (On Windows)

- [ ] All tests passing (run: python run_tests.py)
- [ ] Full workflow tested locally
- [ ] requirements.txt up to date
- [ ] .env file has production configuration
- [ ] credentials.json and token.json generated and working
- [ ] data/clients.csv has all 30 clients
- [ ] Documentation reviewed (README.md, USAGE.md, DEPLOYMENT.md)

## File Upload (Via cPanel or FTP)

- [ ] Upload all files from deploy/ directory to /home/username/email_reports/
- [ ] Verify folder structure:
      - /home/username/email_reports/src/
      - /home/username/email_reports/templates/
      - /home/username/email_reports/data/
      - /home/username/email_reports/logs/
- [ ] Upload .env file separately (use .env.linux as template)
- [ ] Upload credentials.json
- [ ] Upload token.json
- [ ] Upload data/clients.csv

## Server Configuration (Via SSH or cPanel Terminal)

### 1. Update .env File

```bash
cd /home/username/email_reports
nano .env
# Replace 'username' with actual cPanel username in all paths
# Save: Ctrl+O, Exit: Ctrl+X
```

### 2. Set File Permissions

```bash
# Make directories accessible
find /home/username/email_reports -type d -exec chmod 755 {{}} \\;

# Make Python files readable/executable
find /home/username/email_reports -type f -name "*.py" -exec chmod 755 {{}} \\;

# Secure sensitive files
chmod 600 /home/username/email_reports/.env
chmod 600 /home/username/email_reports/credentials.json
chmod 600 /home/username/email_reports/token.json
chmod 600 /home/username/email_reports/data/clients.csv

# Verify
ls -la /home/username/email_reports/
```

### 3. Create Virtual Environment

```bash
cd /home/username/email_reports

# Create venv
python3 -m venv venv

# Activate
source venv/bin/activate

# Upgrade pip
pip install --upgrade pip

# Install dependencies
pip install -r requirements.txt

# Verify
pip list
```

### 4. Test Components

```bash
# Activate venv (if not already)
source venv/bin/activate

# Test 1: Python version
python3 --version
# Should show: Python 3.8+

# Test 2: Import test
python3 -c "import pdfplumber, jinja2, gspread; print('✓ Imports successful')"

# Test 3: Gmail authentication
python3 -c "from src.gmail_reader import GmailReader; r = GmailReader(); print('✓ Gmail authenticated')"

# Test 4: Client database
python3 -c "from src.client_database import ClientDatabase; db = ClientDatabase('data/clients.csv'); print(f'✓ Loaded {{len(db.clients)}} clients')"

# Test 5: Main script
python3 main.py --help
```

### 5. Test Workflow

```bash
# If test PDFs available:
python3 main.py --process-pdfs

# Or extract from Drive:
python3 main.py --extract-from-drive --month "October"

# Check logs
tail -f logs/email_reports_$(date +%Y-%m-%d).log
```

## Cron Job Setup (Monthly Automation)

### Via cPanel Cron Jobs

1. Log in to cPanel
2. Navigate to: Advanced → Cron Jobs
3. Add cron job:
   - **Minute:** 0
   - **Hour:** 9
   - **Day:** 1
   - **Month:** * (every month)
   - **Weekday:** * (any day)
   - **Command:**
     ```
     cd /home/username/email_reports && source venv/bin/activate && python3 main.py --full >> logs/cron_$(date +\\%Y\\%m\\%d).log 2>&1
     ```

### Via SSH (crontab)

```bash
crontab -e

# Add this line:
0 9 1 * * cd /home/username/email_reports && source venv/bin/activate && python3 main.py --full >> logs/cron_$(date +\\%Y\\%m\\%d).log 2>&1

# Save and exit
# Verify:
crontab -l
```

## Post-Deployment Testing

- [ ] Run verification script: ./verify_deployment.sh
- [ ] Test manual PDF extraction
- [ ] Test email generation
- [ ] Test approval workflow (Google Sheets)
- [ ] Test draft creation
- [ ] Send test email to yourself
- [ ] Verify logs are written correctly
- [ ] Check cron job will execute (wait for scheduled time or test with */5 * * * *)

## Parallel Testing (First Month)

- [ ] Run new system on Linux server
- [ ] Keep Relevance AI running as backup
- [ ] Compare outputs
- [ ] Verify all 30 emails generated
- [ ] Document any issues

## Cutover (After Successful Testing)

- [ ] Disable Relevance AI
- [ ] Disable Make.com scenario (if applicable)
- [ ] Update documentation
- [ ] Notify team of new system
- [ ] Monitor for first 3 months

---

**Deployment Date:** _______________
**Deployed By:** _______________
**Server:** _______________
**Status:** _______________

""".format(timestamp=self.timestamp)

        checklist_file = self.deploy_dir / "DEPLOYMENT_CHECKLIST.md"
        with open(checklist_file, 'w', encoding='utf-8') as f:
            f.write(checklist_content)

        print(f"   ✓ Created DEPLOYMENT_CHECKLIST.md")

    def _create_verification_script(self):
        """Create Linux verification script."""
        print(f"\n🔍 Creating verification script...")

        verify_script = """#!/bin/bash

##############################################################################
# Email Reports - Deployment Verification Script
# Run this script after deploying to Linux server to verify setup
##############################################################################

echo "=================================="
echo "Email Reports - Deployment Verification"
echo "=================================="

ERRORS=0
WARNINGS=0

# Check 1: Python version
echo ""
echo "1. Checking Python version..."
PYTHON_VERSION=$(python3 --version 2>&1)
if [[ $PYTHON_VERSION =~ "3.8" ]] || [[ $PYTHON_VERSION =~ "3.9" ]] || [[ $PYTHON_VERSION =~ "3.10" ]] || [[ $PYTHON_VERSION =~ "3.11" ]] || [[ $PYTHON_VERSION =~ "3.12" ]]; then
    echo "   ✓ $PYTHON_VERSION"
else
    echo "   ✗ Python 3.8+ required, found: $PYTHON_VERSION"
    ERRORS=$((ERRORS + 1))
fi

# Check 2: Virtual environment
echo ""
echo "2. Checking virtual environment..."
if [ -d "venv" ]; then
    echo "   ✓ venv/ directory exists"
    if [ -f "venv/bin/activate" ]; then
        echo "   ✓ venv/bin/activate exists"
    else
        echo "   ✗ venv/bin/activate not found"
        ERRORS=$((ERRORS + 1))
    fi
else
    echo "   ✗ venv/ directory not found"
    ERRORS=$((ERRORS + 1))
fi

# Check 3: Required files
echo ""
echo "3. Checking required files..."
REQUIRED_FILES=(".env" "credentials.json" "token.json" "data/clients.csv" "main.py" "requirements.txt")
for file in "${REQUIRED_FILES[@]}"; do
    if [ -f "$file" ]; then
        echo "   ✓ $file"
    else
        echo "   ✗ Missing: $file"
        ERRORS=$((ERRORS + 1))
    fi
done

# Check 4: Source files
echo ""
echo "4. Checking source files..."
SOURCE_FILES=("src/orchestrator.py" "src/pdf_extractor.py" "src/client_database.py" "src/email_generator.py" "src/gmail_reader.py" "src/gmail_sender.py")
MISSING_SRC=0
for file in "${SOURCE_FILES[@]}"; do
    if [ ! -f "$file" ]; then
        echo "   ✗ Missing: $file"
        MISSING_SRC=$((MISSING_SRC + 1))
    fi
done
if [ $MISSING_SRC -eq 0 ]; then
    echo "   ✓ All source files present"
else
    echo "   ✗ Missing $MISSING_SRC source files"
    ERRORS=$((ERRORS + MISSING_SRC))
fi

# Check 5: .env file paths
echo ""
echo "5. Checking .env configuration..."
if [ -f ".env" ]; then
    if grep -q "c:/Apps" .env || grep -q "C:" .env; then
        echo "   ✗ .env contains Windows paths - MUST UPDATE!"
        ERRORS=$((ERRORS + 1))
    else
        echo "   ✓ .env paths appear to be Linux format"
    fi

    if grep -q "username" .env; then
        echo "   ⚠️  .env contains 'username' placeholder - should be replaced with actual username"
        WARNINGS=$((WARNINGS + 1))
    fi
else
    echo "   ✗ .env file not found"
    ERRORS=$((ERRORS + 1))
fi

# Check 6: File permissions
echo ""
echo "6. Checking file permissions..."
if [ -f ".env" ]; then
    ENV_PERMS=$(stat -c "%a" .env 2>/dev/null || stat -f "%A" .env 2>/dev/null)
    if [ "$ENV_PERMS" == "600" ]; then
        echo "   ✓ .env permissions: $ENV_PERMS (secure)"
    else
        echo "   ⚠️  .env permissions: $ENV_PERMS (should be 600)"
        WARNINGS=$((WARNINGS + 1))
    fi
fi

# Check 7: Directories
echo ""
echo "7. Checking directory structure..."
REQUIRED_DIRS=("src" "templates" "data" "logs" "data/pdfs" "data/email_previews")
for dir in "${REQUIRED_DIRS[@]}"; do
    if [ -d "$dir" ]; then
        echo "   ✓ $dir/"
    else
        echo "   ✗ Missing: $dir/"
        ERRORS=$((ERRORS + 1))
    fi
done

# Check 8: Python dependencies (if venv active)
echo ""
echo "8. Checking Python dependencies..."
if [ -n "$VIRTUAL_ENV" ]; then
    DEPS=("pdfplumber" "jinja2" "gspread" "rapidfuzz" "premailer" "dotenv")
    MISSING_DEPS=0
    for dep in "${DEPS[@]}"; do
        python3 -c "import ${dep}" 2>/dev/null
        if [ $? -eq 0 ]; then
            echo "   ✓ $dep"
        else
            echo "   ✗ Missing: $dep"
            MISSING_DEPS=$((MISSING_DEPS + 1))
        fi
    done
    if [ $MISSING_DEPS -gt 0 ]; then
        echo "   ⚠️  $MISSING_DEPS dependencies missing - run: pip install -r requirements.txt"
        WARNINGS=$((WARNINGS + 1))
    fi
else
    echo "   ⚠️  Virtual environment not active - skipping dependency check"
    echo "   ℹ️  Activate with: source venv/bin/activate"
    WARNINGS=$((WARNINGS + 1))
fi

# Summary
echo ""
echo "=================================="
echo "SUMMARY"
echo "=================================="
if [ $ERRORS -eq 0 ] && [ $WARNINGS -eq 0 ]; then
    echo "✅ Deployment verification PASSED"
    echo "   System is ready for production use."
elif [ $ERRORS -eq 0 ]; then
    echo "⚠️  Deployment verification PASSED WITH WARNINGS"
    echo "   Found $WARNINGS warning(s) - review before proceeding."
else
    echo "❌ Deployment verification FAILED"
    echo "   Errors: $ERRORS"
    echo "   Warnings: $WARNINGS"
    echo "   Fix errors before proceeding."
    exit 1
fi
echo "=================================="
"""

        verify_file = self.deploy_dir / "verify_deployment.sh"
        with open(verify_file, 'w', encoding='utf-8', newline='\n') as f:
            f.write(verify_script)

        print(f"   ✓ Created verify_deployment.sh")
        print(f"   ℹ️  Make executable on server: chmod +x verify_deployment.sh")

    def _generate_deployment_summary(self):
        """Generate deployment summary."""
        print(f"\n📋 Generating deployment summary...")

        # Count files
        total_files = sum(1 for _ in self.deploy_dir.rglob("*") if _.is_file())
        py_files = sum(1 for _ in self.deploy_dir.rglob("*.py"))

        summary = f"""
{'='*80}
DEPLOYMENT PACKAGE READY
{'='*80}

Location: {self.deploy_dir}
Total Files: {total_files}
Python Files: {py_files}
Timestamp: {self.timestamp}

Directory Structure:
{self._get_tree_structure()}

Next Steps:
1. Review files in deploy/ directory
2. Upload to Linux server via cPanel File Manager or FTP
3. Follow DEPLOYMENT_CHECKLIST.md step-by-step
4. Run verify_deployment.sh on server to verify setup

Important Files:
- .env.linux           → Rename to .env on server (update paths!)
- DEPLOYMENT_CHECKLIST.md → Complete deployment checklist
- verify_deployment.sh → Run on server after upload
- requirements.txt     → Install with: pip install -r requirements.txt

Files NOT included (upload separately):
- .env (copy from Windows, update paths)
- credentials.json (copy from Windows)
- token.json (copy from Windows)

{'='*80}
"""

        summary_file = self.deploy_dir / "DEPLOYMENT_SUMMARY.txt"
        with open(summary_file, 'w', encoding='utf-8') as f:
            f.write(summary)

        print(summary)

        # Write summary to console
        print(f"   ✓ Created DEPLOYMENT_SUMMARY.txt")

    def _get_tree_structure(self, max_depth=2):
        """Get directory tree structure."""
        tree_lines = []

        def add_directory(path: Path, prefix="", depth=0):
            if depth > max_depth:
                return

            items = sorted(path.iterdir(), key=lambda x: (not x.is_dir(), x.name))
            for i, item in enumerate(items):
                is_last = i == len(items) - 1
                current_prefix = "└── " if is_last else "├── "
                tree_lines.append(f"{prefix}{current_prefix}{item.name}{'/' if item.is_dir() else ''}")

                if item.is_dir() and depth < max_depth:
                    extension = "    " if is_last else "│   "
                    add_directory(item, prefix + extension, depth + 1)

        tree_lines.append(f"{self.deploy_dir.name}/")
        add_directory(self.deploy_dir, "", 0)

        return "\n".join(tree_lines)


def main():
    """Main entry point."""
    project_root = Path(__file__).parent
    prep = DeploymentPreparation(project_root)
    prep.prepare_deployment_package()


if __name__ == "__main__":
    main()
