"""
File Manager Module.
Handles PDF renaming and archiving operations.
"""

import os
import shutil
from pathlib import Path
from datetime import datetime
from typing import List, Dict
from .logger import get_logger

logger = get_logger('FileManager')


class FileManager:
    """Manages PDF file operations including renaming and archiving."""

    def __init__(self, pdf_storage_path: str):
        """
        Initialize file manager.

        Args:
            pdf_storage_path: Path to PDF storage directory
        """
        self.pdf_storage_path = Path(pdf_storage_path)
        self.archive_path = self.pdf_storage_path / 'archive'
        self.logger = logger

    def rename_pdf_with_month_year(self, pdf_path: str, business_name: str,
                                   report_month: str, report_type: str) -> str:
        """
        Rename PDF to include month and year.

        Args:
            pdf_path: Current PDF file path
            business_name: Client business name
            report_month: Month from report (e.g., "January 2025")
            report_type: 'SEO' or 'Google Ads'

        Returns:
            New PDF file path

        Example:
            Old: "ABC Corporation.pdf"
            New: "ABC Corporation - January 2025 SEO Report.pdf"
        """
        try:
            pdf_file = Path(pdf_path)

            # Clean business name for filename
            clean_name = self._sanitize_filename(business_name)

            # Build new filename
            new_filename = f"{clean_name} - {report_month} {report_type} Report.pdf"
            new_path = pdf_file.parent / new_filename

            # Rename file
            if pdf_file.exists():
                pdf_file.rename(new_path)
                self.logger.info(f"Renamed: {pdf_file.name} -> {new_filename}")
                return str(new_path)
            else:
                self.logger.warning(f"PDF not found: {pdf_path}")
                return pdf_path

        except Exception as e:
            self.logger.error(f"Failed to rename PDF {pdf_path}: {e}")
            return pdf_path

    def archive_previous_month_files(self, current_month: str = None) -> Dict:
        """
        Archive all files from previous months.

        Args:
            current_month: Current month name (e.g., "January").
                         If None, auto-detects current month.

        Returns:
            Dictionary with archive statistics
        """
        from dateutil.relativedelta import relativedelta

        try:
            # Determine current month
            if not current_month:
                current_month = datetime.now().strftime('%B')

            self.logger.info(f"Archiving files not from {current_month}")

            # Create archive directory structure
            archive_root = self.pdf_storage_path / 'archive'
            archive_root.mkdir(exist_ok=True)

            # Find all PDF files in storage (not in archive)
            pdf_files = [f for f in self.pdf_storage_path.glob('*.pdf')]

            archived_count = 0
            skipped_count = 0
            error_count = 0

            for pdf_file in pdf_files:
                # Check if filename contains current month
                if current_month.lower() in pdf_file.name.lower():
                    skipped_count += 1
                    continue

                # Determine which month folder to archive to
                archive_month = self._extract_month_from_filename(pdf_file.name)

                if archive_month:
                    # Create month-specific archive folder
                    month_archive = archive_root / archive_month
                    month_archive.mkdir(exist_ok=True)

                    # Move file to archive
                    destination = month_archive / pdf_file.name
                    try:
                        shutil.move(str(pdf_file), str(destination))
                        archived_count += 1
                        self.logger.info(f"Archived: {pdf_file.name} -> archive/{archive_month}/")
                    except Exception as e:
                        self.logger.error(f"Failed to archive {pdf_file.name}: {e}")
                        error_count += 1
                else:
                    # No month found, archive to 'unknown' folder
                    unknown_archive = archive_root / 'unknown'
                    unknown_archive.mkdir(exist_ok=True)

                    destination = unknown_archive / pdf_file.name
                    try:
                        shutil.move(str(pdf_file), str(destination))
                        archived_count += 1
                        self.logger.info(f"Archived: {pdf_file.name} -> archive/unknown/")
                    except Exception as e:
                        self.logger.error(f"Failed to archive {pdf_file.name}: {e}")
                        error_count += 1

            # Also archive preview HTML files
            preview_dir = Path('data/email_previews')
            if preview_dir.exists():
                preview_archive = archive_root / 'email_previews'

                if current_month:
                    # Create month-specific preview archive
                    prev_month = (datetime.now() - relativedelta(months=1)).strftime('%B %Y')
                    preview_archive = preview_archive / prev_month

                preview_archive.mkdir(parents=True, exist_ok=True)

                # Move all preview HTML files
                for html_file in preview_dir.glob('*.html'):
                    destination = preview_archive / html_file.name
                    try:
                        shutil.move(str(html_file), str(destination))
                        self.logger.info(f"Archived preview: {html_file.name}")
                    except Exception as e:
                        self.logger.error(f"Failed to archive preview {html_file.name}: {e}")

            return {
                'archived': archived_count,
                'skipped': skipped_count,
                'errors': error_count,
                'current_month': current_month
            }

        except Exception as e:
            self.logger.error(f"Archive operation failed: {e}")
            return {
                'archived': 0,
                'skipped': 0,
                'errors': 0,
                'error': str(e)
            }

    def _extract_month_from_filename(self, filename: str) -> str:
        """
        Extract month name from PDF filename.

        Args:
            filename: PDF filename

        Returns:
            Month name (e.g., "January 2025") or None
        """
        months = [
            'January', 'February', 'March', 'April', 'May', 'June',
            'July', 'August', 'September', 'October', 'November', 'December'
        ]

        # Look for month name in filename
        for month in months:
            if month.lower() in filename.lower():
                # Try to extract year too
                import re
                year_match = re.search(r'20\d{2}', filename)
                if year_match:
                    return f"{month} {year_match.group()}"
                else:
                    return month

        return None

    def _sanitize_filename(self, name: str) -> str:
        """
        Sanitize filename by removing invalid characters.

        Args:
            name: Original filename

        Returns:
            Sanitized filename
        """
        # Remove invalid Windows filename characters
        invalid_chars = '<>:"/\\|?*'
        for char in invalid_chars:
            name = name.replace(char, '')

        # Remove leading/trailing spaces and dots
        name = name.strip('. ')

        return name
