"""
Main Orchestrator Module.
Coordinates the complete email report workflow.
"""

import os
from pathlib import Path
from typing import Dict, List
from dotenv import load_dotenv
from .logger import get_logger, ReportLogger
from .pdf_extractor import PDFExtractor
from .client_database import ClientDatabase
from .email_generator import EmailGenerator
from .gmail_reader import GmailReader
from .gmail_sender import GmailSender
from .drive_reader import DriveReader
from .approval_tracker import ApprovalTracker
from .file_manager import FileManager
from .web_preview import WebPreviewGenerator

logger = get_logger('Orchestrator')


class ReportOrchestrator:
    """Main orchestrator for the email reporting system."""

    def __init__(self, env_file: str = '.env'):
        """
        Initialize the orchestrator.

        Args:
            env_file: Path to .env configuration file
        """
        # Load environment variables
        load_dotenv(env_file)

        # Initialize logger
        log_level = os.getenv('LOG_LEVEL', 'INFO')
        self.report_logger = ReportLogger(log_dir='logs', log_level=log_level)
        self.logger = logger

        self.logger.info("Initializing Email Reports Orchestrator")

        # Load configuration
        self.config = self._load_config()

        # Initialize modules
        self.pdf_extractor = PDFExtractor()
        self.client_db = ClientDatabase(
            csv_path=self.config['CLIENT_DATABASE_PATH'],
            fuzzy_threshold=int(self.config.get('FUZZY_MATCH_THRESHOLD', 85))
        )
        self.email_generator = EmailGenerator(
            template_path=self.config['TEMPLATE_PATH'],
            config=self.config
        )
        self.approval_tracker = ApprovalTracker(credentials_path='token.json')

        # Gmail and Drive modules (initialized on demand)
        self.gmail_reader = None
        self.gmail_sender = None
        self.drive_reader = None

        # File management modules
        self.file_manager = FileManager(pdf_storage_path=self.config['PDF_STORAGE_PATH'])
        self.web_preview = WebPreviewGenerator(output_dir='output')

        self.logger.info("Orchestrator initialized successfully")

    def _load_config(self) -> Dict:
        """Load configuration from environment variables."""
        config = {
            'GMAIL_SENDER_EMAIL': os.getenv('GMAIL_SENDER_EMAIL'),
            'LOOKER_STUDIO_SENDER': os.getenv('LOOKER_STUDIO_SENDER'),
            'LOOKER_STUDIO_SENDERS': os.getenv('LOOKER_STUDIO_SENDERS'),
            'CLIENT_DATABASE_PATH': os.getenv('CLIENT_DATABASE_PATH'),
            'PDF_STORAGE_PATH': os.getenv('PDF_STORAGE_PATH'),
            'TEMPLATE_PATH': os.getenv('TEMPLATE_PATH'),
            'APPROVAL_TRACKING_PATH': os.getenv('APPROVAL_TRACKING_PATH'),
            'FUZZY_MATCH_THRESHOLD': os.getenv('FUZZY_MATCH_THRESHOLD', '85'),
            'MAX_PDFS_PER_RUN': os.getenv('MAX_PDFS_PER_RUN', '50'),
            'DRIVE_SEO_FOLDER_ID': os.getenv('DRIVE_SEO_FOLDER_ID'),
            'DRIVE_SEM_FOLDER_ID': os.getenv('DRIVE_SEM_FOLDER_ID'),
            'AGENCY_NAME': os.getenv('AGENCY_NAME'),
            'AGENCY_EMAIL': os.getenv('AGENCY_EMAIL'),
            'AGENCY_PHONE': os.getenv('AGENCY_PHONE'),
            'AGENCY_WEBSITE': os.getenv('AGENCY_WEBSITE'),
            'STANDARD_SEO_PARAGRAPH': os.getenv('STANDARD_SEO_PARAGRAPH'),
            'STANDARD_SEM_PARAGRAPH': os.getenv('STANDARD_SEM_PARAGRAPH'),
            'STANDARD_CLOSING_PARAGRAPH': os.getenv('STANDARD_CLOSING_PARAGRAPH'),
        }

        self.logger.info("Configuration loaded from environment")
        return config

    def run_full_workflow(self, extract_from_gmail: bool = False, extract_from_drive: bool = True,
                         create_drafts: bool = True, month: str = None):
        """
        Run the complete workflow from PDF extraction to draft creation.

        Args:
            extract_from_gmail: Whether to extract PDFs from Gmail
            extract_from_drive: Whether to extract PDFs from Google Drive (default)
            create_drafts: Whether to create Gmail drafts for approved emails
            month: Month to process (auto-detects if None)
        """
        self.logger.info("="*60)
        self.logger.info("STARTING FULL WORKFLOW")
        self.logger.info("="*60)

        try:
            # Step 0: Archive previous month's files if running for new month
            from datetime import datetime
            current_month = datetime.now().strftime('%B')
            self.logger.info(f"Archiving old files (keeping {current_month} files)")
            archive_stats = self.file_manager.archive_previous_month_files(current_month)
            self.logger.info(f"Archived {archive_stats['archived']} files, skipped {archive_stats['skipped']}")

            # Step 1: Extract PDFs (from Gmail, Drive, or use existing)
            pdf_files = []
            if extract_from_drive:
                pdf_files = self.extract_pdfs_from_drive(month=month)
            elif extract_from_gmail:
                pdf_files = self.extract_pdfs_from_gmail()
            else:
                # Use PDFs already in storage directory
                pdf_storage = Path(self.config['PDF_STORAGE_PATH'])
                pdf_files = list(pdf_storage.glob('*.pdf'))
                self.logger.info(f"Found {len(pdf_files)} PDFs in storage directory")

            if not pdf_files:
                self.logger.warning("No PDFs found to process")
                return

            # Step 2: Process each PDF
            self.logger.info(f"Processing {len(pdf_files)} PDF files")
            processed_data = self.process_pdfs(pdf_files)

            # Step 3: Generate emails
            self.logger.info("Generating emails")
            email_data = self.generate_emails(processed_data)

            # Step 3.5: Rename PDFs to include month and year
            self.logger.info("Renaming PDFs with month and year")
            for email in email_data:
                if email.get('pdf_path'):
                    new_path = self.file_manager.rename_pdf_with_month_year(
                        pdf_path=email['pdf_path'],
                        business_name=email.get('business_name', 'Unknown'),
                        report_month=email.get('report_month', 'Unknown'),
                        report_type=email.get('report_type', 'SEO')
                    )
                    email['pdf_path'] = new_path

            # Step 4: Save HTML previews to disk
            preview_dir = Path('data/email_previews')
            preview_dir.mkdir(parents=True, exist_ok=True)

            for email in email_data:
                # Save HTML preview
                business_name = email.get('business_name', 'Unknown').replace(' ', '_')
                preview_filename = f"{business_name}_preview.html"
                preview_path = preview_dir / preview_filename

                with open(preview_path, 'w', encoding='utf-8') as f:
                    f.write(email.get('html_body', ''))

                email['preview_path'] = str(preview_path)

            # Step 5: Create Google Sheets approval tracker
            self.logger.info("Creating approval sheet in Google Sheets")
            from datetime import datetime
            sheet_name = f"Email Approvals - {datetime.now().strftime('%B %Y')}"

            # Prepare email data for approval sheet
            approval_data = []
            for email in email_data:
                approval_data.append({
                    'client_name': email.get('recipient_name', 'Unknown'),
                    'business_name': email.get('business_name', 'Unknown'),
                    'recipient_email': email.get('recipient_email', ''),
                    'report_type': email.get('report_type', 'SEO'),
                    'extraction_errors': '; '.join(email.get('extraction_errors', [])),
                    'subject': email.get('subject', ''),
                    'html_body': email.get('html_body', ''),
                    'text_body': email.get('text_body', ''),
                    'preview_path': email.get('preview_path', '')
                })

            # Authenticate and create approval sheet
            self.approval_tracker.authenticate()
            sheet_id = self.approval_tracker.create_approval_sheet(sheet_name, approval_data)
            sheet_url = f"https://docs.google.com/spreadsheets/d/{sheet_id}"

            # Save email data and sheet ID to file for --create-drafts command (do this BEFORE returning)
            import json
            cache_data = {
                'sheet_id': sheet_id,
                'sheet_url': sheet_url,
                'emails': email_data
            }
            email_cache_path = Path('data/last_generated_emails.json')
            with open(email_cache_path, 'w', encoding='utf-8') as f:
                json.dump(cache_data, f, indent=2)
            self.logger.info(f"Saved email data to {email_cache_path}")

            # Step 6: Generate web preview page
            self.logger.info("Generating web preview page")
            from datetime import datetime
            report_month = datetime.now().strftime('%B %Y')
            web_preview_path = self.web_preview.generate_preview_page(email_data, report_month)
            self.logger.info(f"Web preview generated: {web_preview_path}")

            print()
            print("="*70)
            print("  APPROVAL REQUIRED")
            print("="*70)
            print()
            print(f"[SHEET] Your approval sheet:")
            print(f"   {sheet_url}")
            print()
            print(f"[WEB] Web preview page:")
            print(f"   file:///{os.path.abspath(web_preview_path).replace(chr(92), '/')}")
            print()
            print("NEXT STEPS:")
            print("1. Open the Google Sheet (link above) OR")
            print("   Open the web preview page (link above) for visual review")
            print("2. Review each email's details")
            print("3. Change Status to 'Approved' for emails ready to send")
            print("4. Add notes for any that need revision")
            print()

            if not create_drafts:
                print("When done, run:")
                print("  python main.py --create-drafts")
                print()
                self.logger.info(f"Approval sheet created: {sheet_url}")
                return  # Exit here if not creating drafts immediately

            # Wait for approval before continuing
            input("Press Enter when you've finished reviewing and approving emails...")
            print()

            # Step 5: Create drafts for approved emails (if requested)
            if create_drafts:
                approved_count = self.create_drafts_for_approved()
                self.logger.info(f"Created {approved_count} Gmail drafts")

            # Print summary
            summary = self.approval_tracker.get_approval_summary()
            self.logger.info("="*60)
            self.logger.info("WORKFLOW SUMMARY")
            self.logger.info("="*60)
            self.logger.info(f"Total PDFs processed: {len(pdf_files)}")
            self.logger.info(f"Emails generated: {len(email_data)}")
            self.logger.info(f"Approved: {summary['approved']}")
            self.logger.info(f"Pending: {summary['pending']}")
            self.logger.info(f"Needs Revision: {summary['needs_revision']}")
            self.logger.info("="*60)

            # Store email_data for later draft creation
            self.last_generated_emails = email_data

        except Exception as e:
            self.logger.error(f"Workflow failed: {str(e)}")
            raise

    def extract_pdfs_from_drive(self, folder_id: str = None, month: str = None) -> List[str]:
        """
        Extract PDFs from Google Drive folder with automatic month detection.

        Args:
            folder_id: Google Drive folder ID (or full URL).
                      If None, uses DRIVE_SEO_FOLDER_ID and DRIVE_SEM_FOLDER_ID from config.
            month: Month name to process (e.g., "January", "February").
                   If None, automatically detects previous month.

        Returns:
            List of paths to downloaded PDF files
        """
        from datetime import datetime
        from dateutil.relativedelta import relativedelta

        if not self.drive_reader:
            self.drive_reader = DriveReader()

        # Determine which month to process
        if not month:
            # Auto-detect: previous month
            last_month = datetime.now() - relativedelta(months=1)
            month = last_month.strftime('%B')  # "January", "February", etc.
            self.logger.info(f"Auto-detected month: {month}")

        print(f"\nProcessing reports for: {month}")

        pdf_files = []
        pdf_storage = self.config['PDF_STORAGE_PATH']
        max_files = int(self.config.get('MAX_PDFS_PER_RUN', 50))

        self.logger.info(f"PDF storage path: {pdf_storage}")
        self.logger.info(f"Max files per run: {max_files}")

        # Process SEO folder
        seo_folder_id = self.config.get('DRIVE_SEO_FOLDER_ID')
        self.logger.info(f"SEO folder ID from config: '{seo_folder_id}'")
        if seo_folder_id:
            self.logger.info(f"Processing SEO folder: {seo_folder_id}")
            print(f"  - Downloading SEO reports from {month} folder...")

            # Find month subfolder
            month_folder_id = self.drive_reader.find_subfolder_by_name(seo_folder_id, month)

            if month_folder_id:
                seo_files = self.drive_reader.download_all_pdfs(
                    folder_id=month_folder_id,
                    output_dir=pdf_storage,
                    max_files=max_files
                )
                pdf_files.extend(seo_files)
                print(f"     [OK] Downloaded {len(seo_files)} SEO reports")
            else:
                print(f"     [WARNING] No '{month}' folder found in SEO reports")

        # Process SEM (Google Ads) folder
        sem_folder_id = self.config.get('DRIVE_SEM_FOLDER_ID')
        self.logger.info(f"SEM folder ID from config: '{sem_folder_id}'")
        if sem_folder_id:
            self.logger.info(f"Processing SEM folder: {sem_folder_id}")
            print(f"  - Downloading Google Ads reports from {month} folder...")

            # Find month subfolder
            month_folder_id = self.drive_reader.find_subfolder_by_name(sem_folder_id, month)

            if month_folder_id:
                sem_files = self.drive_reader.download_all_pdfs(
                    folder_id=month_folder_id,
                    output_dir=pdf_storage,
                    max_files=max_files
                )
                pdf_files.extend(sem_files)
                print(f"     [OK] Downloaded {len(sem_files)} Google Ads reports")
            else:
                print(f"     [WARNING] No '{month}' folder found in Google Ads reports")

        print(f"\nTotal: Downloaded {len(pdf_files)} PDF files")
        return pdf_files

    def extract_pdfs_from_gmail(self) -> List[str]:
        """Extract PDFs from Gmail."""
        if not self.gmail_reader:
            self.gmail_reader = GmailReader()

        looker_sender = self.config['LOOKER_STUDIO_SENDER']
        pdf_storage = self.config['PDF_STORAGE_PATH']

        self.logger.info(f"Extracting PDFs from emails sent by {looker_sender}")

        pdf_files = self.gmail_reader.extract_all_pdfs(
            sender_email=looker_sender,
            output_dir=pdf_storage,
            mark_as_read=False  # Don't mark as read automatically
        )

        return pdf_files

    def process_pdfs(self, pdf_files: List[str]) -> List[Dict]:
        """
        Process PDF files and extract data.

        Args:
            pdf_files: List of PDF file paths

        Returns:
            List of processed data dictionaries
        """
        processed_data = []

        for idx, pdf_path in enumerate(pdf_files, 1):
            self.logger.info(f"Processing PDF {idx}/{len(pdf_files)}: {Path(pdf_path).name}")

            # Extract data from PDF
            extracted = self.pdf_extractor.extract_report_data(pdf_path)

            # Match to client
            client = None
            if extracted['business_name']:
                client = self.client_db.find_client(extracted['business_name'])

            if not client:
                self.logger.warning(f"No client match found for: {extracted['business_name']}")
                extracted['extraction_errors'].append("No matching client in database")

            processed_data.append({
                'pdf_path': pdf_path,
                'extracted': extracted,
                'client': client
            })

        return processed_data

    def generate_emails(self, processed_data: List[Dict]) -> List[Dict]:
        """
        Generate emails for processed data.

        Args:
            processed_data: List of processed PDF data

        Returns:
            List of email data dictionaries
        """
        email_data = []

        for data in processed_data:
            if not data['client']:
                self.logger.warning(f"Skipping email generation for {data['extracted']['business_name']} - no client match")
                continue

            try:
                email = self.email_generator.generate_email(
                    client_data=data['client'],
                    extracted_data=data['extracted']
                )

                email['pdf_path'] = data['pdf_path']
                email['client_id'] = data['client'].get('ClientID', 'Unknown')
                email['business_name'] = data['client'].get('Client-Name', 'Unknown')
                email['extraction_errors'] = data['extracted']['extraction_errors']
                email['report_month'] = data['extracted'].get('report_month', '')

                email_data.append(email)

            except Exception as e:
                client_name = data['client'].get('Client-Name', 'Unknown')
                self.logger.error(f"Failed to generate email for {client_name}: {str(e)}")
                continue

        return email_data

    def create_drafts_for_approved(self) -> int:
        """
        Create Gmail drafts for all approved emails.

        Returns:
            Number of drafts created
        """
        import json

        if not self.gmail_sender:
            self.gmail_sender = GmailSender()  # Authenticates in __init__

        # Load email data and sheet ID from cache if not in memory
        if not hasattr(self, 'last_generated_emails'):
            email_cache_path = Path('data/last_generated_emails.json')

            if not email_cache_path.exists():
                self.logger.error("No email data available. Run workflow first with --full")
                raise ValueError("No email data available - run 'python main.py --full' first")

            with open(email_cache_path, 'r', encoding='utf-8') as f:
                cache_data = json.load(f)

            # Handle both old format (list) and new format (dict)
            if isinstance(cache_data, list):
                self.last_generated_emails = cache_data
                sheet_id = None
            else:
                self.last_generated_emails = cache_data['emails']
                sheet_id = cache_data.get('sheet_id')

            self.logger.info(f"Loaded {len(self.last_generated_emails)} emails from cache")

        # Check for web approvals file first (priority over Google Sheets)
        web_approvals_path = Path('data/web_approvals.json')
        approved_clients = None
        approved_businesses = []  # Initialize for later use

        if web_approvals_path.exists():
            self.logger.info("Found web approvals file - using web preview approvals")
            print("\n[INFO] Using approvals from web preview (data/web_approvals.json)")

            with open(web_approvals_path, 'r', encoding='utf-8') as f:
                web_approvals = json.load(f)

            # Get list of approved emails (format: "BusinessName_ReportType")
            approved_email_keys = [
                key for key, data in web_approvals.items()
                if data.get('status') == 'approved'
            ]

            # Map approved keys to recipient names for filtering
            approved_clients = []
            for email in self.last_generated_emails:
                # Replace spaces with underscores to match JavaScript format (web_preview does this for safe_id)
                business_name = email.get('business_name', '').replace(' ', '_')
                report_type = email.get('report_type', '').replace(' ', '_')
                email_key = f"{business_name}_{report_type}"
                if email_key in approved_email_keys:
                    approved_clients.append(email.get('recipient_name'))

            # Store web approvals for later use (for extra content)
            self._web_approvals = web_approvals

            self.logger.info(f"Found {len(approved_clients)} approved emails from web preview")

        else:
            # Fallback to Google Sheets
            self.logger.info("No web approvals file found - checking Google Sheets")
            print("\n[INFO] Checking Google Sheets for approvals...")

            # Authenticate with approval tracker and get approved clients
            self.approval_tracker.authenticate()

            # Open the existing approval sheet if we have a sheet_id
            if sheet_id:
                self.approval_tracker.open_existing_sheet(sheet_id)
                self.logger.info(f"Opened approval sheet: {sheet_id}")

            approved_clients = self.approval_tracker.get_approved_clients()

        if not approved_clients:
            self.logger.info("No approved emails to create drafts for")
            print("\n[WARNING] No emails were approved.")
            print("  - Either approve emails in the web preview and save web_approvals.json to data/")
            print("  - Or approve emails in the Google Sheet")
            return 0

        self.logger.info(f"Found {len(approved_clients)} approved emails")
        print(f"\nCreating Gmail drafts for {len(approved_clients)} approved emails...")
        print()

        # Filter to approved emails only
        # When using web approvals, match by business_name + report_type
        # When using sheets, match by recipient_name
        if web_approvals_path.exists():
            # Web approvals: match by business name + report type
            # Note: web_preview replaces ALL spaces with underscores in safe_id
            approved_emails = []
            for email in self.last_generated_emails:
                business_name = email.get('business_name', '').replace(' ', '_')
                report_type = email.get('report_type', '').replace(' ', '_')
                email_key = f"{business_name}_{report_type}"
                if email_key in approved_email_keys:
                    approved_emails.append(email)
        else:
            # Google Sheets: match by recipient name
            approved_emails = [
                email for email in self.last_generated_emails
                if email.get('recipient_name') in approved_clients
            ]

        draft_count = 0
        draft_errors = []

        for email in approved_emails:
            try:
                self.logger.info(f"Creating draft for: {email['recipient_name']}")

                # Check if there's extra content for this email from web approvals
                # Replace spaces with underscores to match JavaScript format (web_preview does this for safe_id)
                business_name_key = email.get('business_name', '').replace(' ', '_')
                report_type = email.get('report_type', '').replace(' ', '_')
                email_key = f"{business_name_key}_{report_type}"

                # Check for extra content and inject it right after the opening line
                extra_content = None
                if hasattr(self, '_web_approvals') and email_key in self._web_approvals:
                    approval_data = self._web_approvals[email_key]
                    extra_content_text = approval_data.get('extraContent')

                    if extra_content_text:
                        self.logger.info(f"Adding extra content for {email_key}")
                        print(f"  [EXTRA CONTENT] Adding custom content for {email_key}")

                        # Inject extra content right after opening paragraph
                        html_body = email['html_body']

                        # For SEO reports
                        if 'Please see the data below for' in html_body:
                            marker = '</p>'  # Close of opening paragraph
                            parts = html_body.split(marker, 1)
                            if len(parts) == 2:
                                extra_html = f'\n\n    <p style="margin-bottom: 15px;">{extra_content_text}</p>'
                                html_body = parts[0] + marker + extra_html + parts[1]
                                email['html_body'] = html_body
                        # For Google Ads reports
                        elif 'Please see attached report for' in html_body:
                            marker = '</p>'  # Close of opening paragraph
                            parts = html_body.split(marker, 1)
                            if len(parts) == 2:
                                extra_html = f'\n\n    <p style="margin-bottom: 15px;">{extra_content_text}</p>'
                                html_body = parts[0] + marker + extra_html + parts[1]
                                email['html_body'] = html_body

                # Create draft with PDF attachment
                self.gmail_sender.create_draft(
                    recipient=email['recipient_email'],
                    subject=email['subject'],
                    html_body=email['html_body'],
                    text_body=email['text_body'],
                    attachment_path=email['pdf_path']
                )

                draft_count += 1
                print(f"  [OK] {email['recipient_name']} ({email['business_name']})")

            except Exception as e:
                error_msg = f"Failed to create draft for {email['recipient_name']}: {e}"
                self.logger.error(error_msg)
                draft_errors.append(error_msg)
                print(f"  [FAIL] {email['recipient_name']}: {e}")

        print()
        print("="*70)
        print(f"[OK] Created {draft_count} Gmail drafts")
        if draft_errors:
            print(f"[FAIL] {len(draft_errors)} drafts failed")
        print()
        print("NEXT STEPS:")
        print("1. Open Gmail and go to Drafts folder")
        print("2. Review each draft one final time")
        print("3. Send emails manually, spacing them out (5-10 minutes apart)")
        print()
        print("="*70)

        return draft_count

    def approve_all_pending(self):
        """Approve all pending emails (convenience method for testing)."""
        self.logger.warning("approve_all_pending not implemented for Google Sheets tracker")
        print("\n[WARNING] Auto-approval not supported with Google Sheets tracker.")
        print("Please manually approve emails in the Google Sheet.")
