"""
Web Preview Generator Module.
Creates HTML preview pages for client reports.
"""

import os
from pathlib import Path
from typing import List, Dict
from datetime import datetime
from .logger import get_logger

logger = get_logger('WebPreview')


class WebPreviewGenerator:
    """Generates HTML preview webpage for client reports."""

    def __init__(self, output_dir: str = 'output'):
        """
        Initialize web preview generator.

        Args:
            output_dir: Directory for generated preview pages
        """
        self.output_dir = Path(output_dir)
        self.output_dir.mkdir(parents=True, exist_ok=True)
        self.logger = logger

    def generate_preview_page(self, email_data: List[Dict], month: str = None) -> str:
        """
        Generate HTML preview page for all client reports.

        Args:
            email_data: List of email data dictionaries
            month: Report month (e.g., "January 2025")

        Returns:
            Path to generated HTML file
        """
        try:
            if not month:
                month = datetime.now().strftime('%B %Y')

            # Separate emails by type
            seo_emails = [e for e in email_data if e.get('report_type') == 'SEO']
            sem_emails = [e for e in email_data if e.get('report_type') == 'Google Ads']

            # Generate HTML
            html = self._generate_html(seo_emails, sem_emails, month)

            # Save to file
            filename = f"email_preview_{month.replace(' ', '_')}.html"
            output_path = self.output_dir / filename

            with open(output_path, 'w', encoding='utf-8') as f:
                f.write(html)

            self.logger.info(f"Generated preview page: {output_path}")
            return str(output_path)

        except Exception as e:
            self.logger.error(f"Failed to generate preview page: {e}")
            raise

    def _generate_html(self, seo_emails: List[Dict], sem_emails: List[Dict], month: str) -> str:
        """Generate complete HTML page."""

        all_emails = seo_emails + sem_emails

        html = f"""<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Reports Preview - {month}</title>
    <style>
        * {{
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }}

        body {{
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }}

        .container {{
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
        }}

        header {{
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 20px;
            text-align: center;
            border-radius: 12px;
            margin-bottom: 30px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }}

        header h1 {{
            font-size: 2.5em;
            margin-bottom: 10px;
        }}

        header p {{
            font-size: 1.2em;
            opacity: 0.9;
        }}

        .stats {{
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }}

        .stat-card {{
            background: white;
            padding: 25px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            text-align: center;
        }}

        .stat-card .number {{
            font-size: 3em;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 5px;
        }}

        .stat-card .label {{
            color: #666;
            font-size: 0.9em;
            text-transform: uppercase;
            letter-spacing: 1px;
        }}

        /* Controls Section */
        .controls {{
            background: white;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            align-items: center;
        }}

        .search-box {{
            flex: 1;
            min-width: 250px;
            position: relative;
        }}

        .search-box input {{
            width: 100%;
            padding: 12px 40px 12px 15px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 1em;
            transition: border-color 0.2s;
        }}

        .search-box input:focus {{
            outline: none;
            border-color: #667eea;
        }}

        .search-box .search-icon {{
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            color: #999;
            font-size: 1.2em;
        }}

        .filter-group {{
            display: flex;
            gap: 10px;
            align-items: center;
        }}

        .filter-btn {{
            padding: 10px 20px;
            border: 2px solid #e9ecef;
            background: white;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.2s;
            font-size: 0.95em;
        }}

        .filter-btn:hover {{
            border-color: #667eea;
            color: #667eea;
        }}

        .filter-btn.active {{
            background: #667eea;
            color: white;
            border-color: #667eea;
        }}

        .view-toggle {{
            display: flex;
            gap: 5px;
            background: #f8f9fa;
            padding: 4px;
            border-radius: 8px;
        }}

        .view-btn {{
            padding: 8px 16px;
            border: none;
            background: transparent;
            cursor: pointer;
            border-radius: 6px;
            font-size: 1.2em;
            transition: background 0.2s;
        }}

        .view-btn.active {{
            background: white;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }}

        .results-info {{
            color: #666;
            font-size: 0.95em;
            margin-left: auto;
        }}

        /* Table View */
        .reports-table {{
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            margin-top: 20px;
        }}

        table {{
            width: 100%;
            border-collapse: collapse;
        }}

        thead {{
            background: #f8f9fa;
            position: sticky;
            top: 0;
            z-index: 10;
        }}

        th {{
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #333;
            border-bottom: 2px solid #e9ecef;
            cursor: pointer;
            user-select: none;
            white-space: nowrap;
        }}

        th:hover {{
            background: #e9ecef;
        }}

        th .sort-icon {{
            margin-left: 5px;
            opacity: 0.3;
            font-size: 0.8em;
        }}

        th.sorted .sort-icon {{
            opacity: 1;
        }}

        tbody tr {{
            border-bottom: 1px solid #f0f0f0;
            transition: background 0.2s;
        }}

        tbody tr:hover {{
            background: #f8f9fa;
        }}

        tbody tr.hidden {{
            display: none;
        }}

        td {{
            padding: 15px;
        }}

        .business-name {{
            font-weight: 600;
            color: #333;
        }}

        .badge {{
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.85em;
            font-weight: 600;
        }}

        .badge.seo {{
            background: #dbeafe;
            color: #1e40af;
        }}

        .badge.sem {{
            background: #dcfce7;
            color: #166534;
        }}

        .status-badge {{
            display: inline-block;
            padding: 6px 12px;
            border-radius: 6px;
            font-size: 0.85em;
            font-weight: 600;
            text-transform: uppercase;
        }}

        .status-pending {{
            background: #fef3c7;
            color: #92400e;
        }}

        .status-approved {{
            background: #d1fae5;
            color: #065f46;
        }}

        .status-needs-fix {{
            background: #fee2e2;
            color: #991b1b;
        }}

        .approval-btns {{
            display: flex;
            gap: 6px;
            flex-wrap: wrap;
        }}

        .approve-btn {{
            padding: 6px 12px;
            border: 2px solid #10b981;
            background: white;
            color: #10b981;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.85em;
            font-weight: 600;
            transition: all 0.2s;
        }}

        .approve-btn:hover {{
            background: #10b981;
            color: white;
        }}

        .approve-btn.active {{
            background: #10b981;
            color: white;
        }}

        .fix-btn {{
            padding: 6px 12px;
            border: 2px solid #ef4444;
            background: white;
            color: #ef4444;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.85em;
            font-weight: 600;
            transition: all 0.2s;
        }}

        .fix-btn:hover {{
            background: #ef4444;
            color: white;
        }}

        .fix-btn.active {{
            background: #ef4444;
            color: white;
        }}

        .action-btns {{
            display: flex;
            gap: 8px;
        }}

        .icon-btn {{
            padding: 8px 12px;
            border: 1px solid #e9ecef;
            background: white;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            transition: all 0.2s;
            text-decoration: none;
            display: inline-block;
        }}

        .icon-btn:hover {{
            background: #667eea;
            color: white;
            border-color: #667eea;
        }}

        /* Grid View */
        .client-grid {{
            display: none;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
            padding: 20px;
        }}

        .client-grid.active {{
            display: grid;
        }}

        .client-card {{
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 20px;
            transition: all 0.3s ease;
        }}

        .client-card:hover {{
            transform: translateY(-3px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.2);
            border-color: #667eea;
        }}

        .client-card .business-name {{
            font-size: 1.2em;
            margin-bottom: 8px;
        }}

        .client-card .contact {{
            color: #666;
            font-size: 0.9em;
            margin-bottom: 12px;
        }}

        .client-card .actions {{
            display: flex;
            gap: 10px;
        }}

        .btn {{
            flex: 1;
            padding: 10px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            text-align: center;
            display: inline-block;
            transition: all 0.2s;
        }}

        .btn-primary {{
            background: #667eea;
            color: white;
        }}

        .btn-primary:hover {{
            background: #5568d3;
        }}

        .btn-secondary {{
            background: #48bb78;
            color: white;
        }}

        .btn-secondary:hover {{
            background: #38a169;
        }}

        /* Modal */
        .modal {{
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.7);
            z-index: 1000;
            overflow-y: auto;
        }}

        .modal.active {{
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }}

        .modal-content {{
            background: white;
            width: 100%;
            max-width: 900px;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            position: relative;
            max-height: 90vh;
            overflow-y: auto;
        }}

        .modal-header {{
            background: #667eea;
            color: white;
            padding: 25px;
            border-radius: 12px 12px 0 0;
            position: sticky;
            top: 0;
            z-index: 10;
        }}

        .modal-header h2 {{
            margin-bottom: 5px;
        }}

        .modal-header .subject {{
            opacity: 0.9;
            font-size: 0.95em;
        }}

        .close-btn {{
            position: absolute;
            top: 20px;
            right: 20px;
            background: rgba(255,255,255,0.2);
            border: none;
            color: white;
            font-size: 28px;
            cursor: pointer;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            line-height: 40px;
            text-align: center;
            transition: background 0.2s;
        }}

        .close-btn:hover {{
            background: rgba(255,255,255,0.3);
        }}

        .modal-body {{
            padding: 30px;
        }}

        .email-preview {{
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 30px;
            margin-bottom: 20px;
        }}

        .pdf-link {{
            display: inline-block;
            background: #48bb78;
            color: white;
            padding: 12px 24px;
            border-radius: 6px;
            text-decoration: none;
            font-weight: 600;
            transition: background 0.2s;
        }}

        .pdf-link:hover {{
            background: #38a169;
        }}

        .no-results {{
            text-align: center;
            padding: 60px 20px;
            color: #999;
        }}

        .no-results .icon {{
            font-size: 4em;
            margin-bottom: 20px;
        }}

        @media (max-width: 768px) {{
            header h1 {{
                font-size: 1.8em;
            }}

            .controls {{
                position: static;
            }}

            thead {{
                position: static;
            }}

            table {{
                font-size: 0.9em;
            }}

            th, td {{
                padding: 10px;
            }}

            .action-btns {{
                flex-direction: column;
            }}

            .client-grid {{
                grid-template-columns: 1fr;
            }}

            .modal-content {{
                max-width: 100%;
                margin: 10px;
            }}
        }}
    </style>
</head>
<body>
    <div class="container">
        <header>
            <h1>📊 Email Reports Preview</h1>
            <p>{month}</p>
        </header>

        <div class="stats">
            <div class="stat-card">
                <div class="number">{len(all_emails)}</div>
                <div class="label">Total Reports</div>
            </div>
            <div class="stat-card">
                <div class="number" id="approvedCount" style="color: #10b981;">0</div>
                <div class="label">Approved</div>
            </div>
            <div class="stat-card">
                <div class="number" id="needsFixCount" style="color: #ef4444;">0</div>
                <div class="label">Needs Fix</div>
            </div>
            <div class="stat-card">
                <div class="number" id="pendingCount" style="color: #f59e0b;">{len(all_emails)}</div>
                <div class="label">Pending Review</div>
            </div>
        </div>

        <!-- Approval Actions Bar -->
        <div style="background: white; padding: 15px 20px; border-radius: 12px; margin-bottom: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.05); display: flex; gap: 15px; align-items: center; flex-wrap: wrap;">
            <button onclick="approveAll()" style="padding: 10px 20px; background: #10b981; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; transition: all 0.2s;">
                ✓ Approve All
            </button>
            <button onclick="resetAll()" style="padding: 10px 20px; background: #6b7280; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; transition: all 0.2s;">
                ↺ Reset All
            </button>
            <button onclick="exportApproved()" style="padding: 10px 20px; background: #667eea; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; transition: all 0.2s;">
                ⬇ Download Approvals File
            </button>
            <div style="margin-left: auto; color: #666; font-size: 0.9em;" id="saveStatus"></div>
        </div>

        <!-- Controls -->
        <div class="controls">
            <div class="search-box">
                <input type="text" id="searchInput" placeholder="Search by business name, contact, or email...">
                <span class="search-icon">🔍</span>
            </div>

            <div class="filter-group">
                <label style="margin-right: 10px; font-weight: 600; color: #555;">Type:</label>
                <button class="filter-btn active" data-filter-type="all">All</button>
                <button class="filter-btn" data-filter-type="SEO">SEO Only</button>
                <button class="filter-btn" data-filter-type="Google Ads">Ads Only</button>
            </div>

            <div class="filter-group">
                <label style="margin-right: 10px; font-weight: 600; color: #555;">Client:</label>
                <button class="filter-btn active" data-filter-agency="all">All</button>
                <button class="filter-btn" data-filter-agency="Direct Clients">Direct Clients</button>
                <button class="filter-btn" data-filter-agency="CJU">CJU</button>
                <button class="filter-btn" data-filter-agency="InnovaIT">InnovaIT</button>
                <button class="filter-btn" data-filter-agency="Luna">Luna</button>
                <button class="filter-btn" data-filter-agency="Acurio">Acurio</button>
            </div>

            <div class="view-toggle">
                <button class="view-btn active" data-view="table" title="Table View">☰</button>
                <button class="view-btn" data-view="grid" title="Grid View">⊞</button>
            </div>

            <div class="results-info">
                <span id="resultCount">{len(all_emails)}</span> reports
            </div>
        </div>

        <!-- Table View -->
        <div class="reports-table" id="tableView">
            <table>
                <thead>
                    <tr>
                        <th data-sort="business">Business Name <span class="sort-icon">▼</span></th>
                        <th data-sort="contact">Contact</th>
                        <th data-sort="email">Email</th>
                        <th data-sort="type">Type</th>
                        <th data-sort="status">Status</th>
                        <th>Approval</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="reportsTableBody">
"""

        # Add table rows
        for email in sorted(all_emails, key=lambda x: x.get('business_name', '')):
            html += self._generate_table_row(email)

        html += """
                </tbody>
            </table>
            <div id="noResults" class="no-results" style="display: none;">
                <div class="icon">🔍</div>
                <h3>No reports found</h3>
                <p>Try adjusting your search or filters</p>
            </div>
        </div>

        <!-- Grid View -->
        <div class="client-grid" id="gridView">
"""

        # Add grid cards
        for email in sorted(all_emails, key=lambda x: x.get('business_name', '')):
            html += self._generate_client_card(email)

        html += """
        </div>
    </div>

    <!-- Modal for email preview -->
    <div class="modal" id="emailModal">
        <div class="modal-content">
            <div class="modal-header">
                <button class="close-btn" onclick="closeModal()">&times;</button>
                <h2 id="modalBusinessName"></h2>
                <div class="subject" id="modalSubject"></div>
            </div>
            <div class="modal-body">
                <div class="email-preview" id="modalEmailContent"></div>
                <a href="#" id="modalPdfLink" class="pdf-link" target="_blank">
                    📄 View PDF Report
                </a>
            </div>
        </div>
    </div>

    <!-- Modal for editing email -->
    <div class="modal" id="editModal">
        <div class="modal-content">
            <div class="modal-header">
                <button class="close-btn" onclick="closeEditModal()">&times;</button>
                <h2 id="editModalBusinessName"></h2>
                <div class="subject">Add Extra Content</div>
            </div>
            <div class="modal-body">
                <div style="margin-bottom: 20px;">
                    <label style="display: block; margin-bottom: 10px; font-weight: 600; color: #333;">
                        Extra Content:
                    </label>
                    <textarea
                        id="extraContentInput"
                        style="width: 100%; min-height: 150px; padding: 15px; border: 2px solid #e9ecef; border-radius: 8px; font-family: inherit; font-size: 1em; resize: vertical;"
                        placeholder="Enter additional content to be added to this email..."
                        onpaste="return true;"
                    ></textarea>
                    <div style="margin-top: 10px; color: #666; font-size: 0.9em;">
                        This content will be added after the personalized introduction text.
                    </div>
                </div>
                <div style="display: flex; gap: 10px; justify-content: flex-end;">
                    <button
                        onclick="closeEditModal()"
                        style="padding: 12px 24px; background: #6b7280; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; transition: all 0.2s;"
                    >
                        Cancel
                    </button>
                    <button
                        onclick="saveExtraContent()"
                        style="padding: 12px 24px; background: #10b981; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; transition: all 0.2s;"
                    >
                        Save Content
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Email data
        const emailData = {};

        // Approval tracking
        const STORAGE_KEY = 'email_approvals';
        let approvalStatus = {};

        // Extra content tracking
        const EXTRA_CONTENT_KEY = 'email_extra_content';
        let extraContentData = {};
        let currentEditingClientId = null;

        // Load saved approvals from localStorage
        function loadApprovals() {
            const saved = localStorage.getItem(STORAGE_KEY);
            if (saved) {
                try {
                    approvalStatus = JSON.parse(saved);
                    // Apply saved statuses to UI
                    Object.keys(approvalStatus).forEach(clientId => {
                        const status = approvalStatus[clientId];
                        updateStatusUI(clientId, status);
                    });
                    updateCounts();
                } catch (e) {
                    console.error('Failed to load approvals:', e);
                }
            }

            // Load extra content
            const savedContent = localStorage.getItem(EXTRA_CONTENT_KEY);
            if (savedContent) {
                try {
                    extraContentData = JSON.parse(savedContent);
                    // Update UI to show which emails have extra content
                    Object.keys(extraContentData).forEach(clientId => {
                        updateExtraContentIndicator(clientId);
                    });
                } catch (e) {
                    console.error('Failed to load extra content:', e);
                }
            }
        }

        // Save approvals to localStorage
        function saveApprovals() {
            localStorage.setItem(STORAGE_KEY, JSON.stringify(approvalStatus));
            document.getElementById('saveStatus').textContent = 'Saved ✓';
            setTimeout(() => {
                document.getElementById('saveStatus').textContent = '';
            }, 2000);
        }

        // Set approval status for a client
        function setStatus(clientId, status) {
            const currentStatus = approvalStatus[clientId];

            // If clicking the same status, reset to pending (toggle off)
            if (currentStatus === status) {
                delete approvalStatus[clientId];
                if (approvalStatus[clientId + '_note']) {
                    delete approvalStatus[clientId + '_note'];
                }
                updateStatusUI(clientId, 'pending');
                updateCounts();
                saveApprovals();
                return;
            }

            // If setting to needs-fix, prompt for notes
            if (status === 'needs-fix') {
                const note = prompt('What needs to be fixed?', approvalStatus[clientId + '_note'] || '');
                if (note === null) return; // User cancelled
                approvalStatus[clientId + '_note'] = note;
            } else {
                // Clear notes if approving
                if (approvalStatus[clientId + '_note']) {
                    delete approvalStatus[clientId + '_note'];
                }
            }

            approvalStatus[clientId] = status;
            updateStatusUI(clientId, status);
            updateCounts();
            saveApprovals();
        }

        // Update UI for a specific client status
        function updateStatusUI(clientId, status) {
            const statusBadge = document.getElementById('status_' + clientId);
            const row = document.getElementById('row_' + clientId);
            const approveBtn = row.querySelector('.approve-btn');
            const fixBtn = row.querySelector('.fix-btn');

            if (!statusBadge) return;

            // Update badge
            statusBadge.className = 'status-badge';
            approveBtn.classList.remove('active');
            fixBtn.classList.remove('active');

            if (status === 'approved') {
                statusBadge.classList.add('status-approved');
                statusBadge.textContent = 'Approved';
                approveBtn.classList.add('active');
                row.dataset.status = 'approved';
            } else if (status === 'needs-fix') {
                statusBadge.classList.add('status-needs-fix');
                const hasNote = approvalStatus[clientId + '_note'];
                statusBadge.textContent = hasNote ? 'Needs Fix 📝' : 'Needs Fix';
                statusBadge.title = hasNote ? approvalStatus[clientId + '_note'] : '';
                fixBtn.classList.add('active');
                row.dataset.status = 'needs-fix';
            } else {
                statusBadge.classList.add('status-pending');
                statusBadge.textContent = 'Pending';
                statusBadge.title = '';
                row.dataset.status = 'pending';
            }
        }

        // Update counts in stats
        function updateCounts() {
            let approved = 0, needsFix = 0, pending = 0;

            Object.values(approvalStatus).forEach(status => {
                if (status === 'approved') approved++;
                else if (status === 'needs-fix') needsFix++;
                else pending++;
            });

            // Count rows not in approvalStatus as pending
            const totalRows = document.querySelectorAll('#reportsTableBody tr').length;
            pending = totalRows - approved - needsFix;

            document.getElementById('approvedCount').textContent = approved;
            document.getElementById('needsFixCount').textContent = needsFix;
            document.getElementById('pendingCount').textContent = pending;
        }

        // Approve all emails
        function approveAll() {
            const rows = document.querySelectorAll('#reportsTableBody tr');
            rows.forEach(row => {
                const clientId = row.id.replace('row_', '');
                setStatus(clientId, 'approved');
            });
        }

        // Reset all statuses
        function resetAll() {
            if (confirm('Reset all approvals to Pending?')) {
                approvalStatus = {};
                const rows = document.querySelectorAll('#reportsTableBody tr');
                rows.forEach(row => {
                    const clientId = row.id.replace('row_', '');
                    updateStatusUI(clientId, 'pending');
                });
                updateCounts();
                saveApprovals();
            }
        }

        // Export approved list to JSON file
        async function exportApproved() {
            const exportData = {};

            // Export all statuses (approved, needs-fix, pending) with notes and extra content
            Object.keys(approvalStatus).forEach(clientId => {
                // Skip note entries (they're stored separately)
                if (clientId.endsWith('_note')) return;

                const email = emailData[clientId];
                if (email) {
                    const statusData = {
                        status: approvalStatus[clientId],
                        clientId: clientId,
                        businessName: email.businessName
                    };

                    // Include notes if this client has needs-fix status
                    if (approvalStatus[clientId] === 'needs-fix' && approvalStatus[clientId + '_note']) {
                        statusData.notes = approvalStatus[clientId + '_note'];
                    }

                    // Include extra content if available
                    if (extraContentData[clientId]) {
                        statusData.extraContent = extraContentData[clientId];
                    }

                    // Use clientId as key (includes report type) instead of just businessName
                    exportData[clientId] = statusData;
                }
            });

            const jsonStr = JSON.stringify(exportData, null, 2);

            // Try to use File System Access API for direct save
            try {
                if ('showSaveFilePicker' in window) {
                    const handle = await window.showSaveFilePicker({
                        suggestedName: 'web_approvals.json',
                        types: [{
                            description: 'JSON Files',
                            accept: { 'application/json': ['.json'] }
                        }]
                    });
                    const writable = await handle.createWritable();
                    await writable.write(jsonStr);
                    await writable.close();

                    const approvedCount = Object.keys(exportData).filter(k => exportData[k].status === 'approved').length;
                    document.getElementById('saveStatus').textContent = `Saved! ${approvedCount} approved`;
                } else {
                    throw new Error('File System Access API not supported');
                }
            } catch (err) {
                // Fallback to regular download
                const blob = new Blob([jsonStr], { type: 'application/json' });
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = 'web_approvals.json';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                URL.revokeObjectURL(url);

                const approvedCount = Object.keys(exportData).filter(k => exportData[k].status === 'approved').length;
                document.getElementById('saveStatus').textContent = `Downloaded! ${approvedCount} approved (Save to: data/)`;
            }

            setTimeout(() => {
                document.getElementById('saveStatus').textContent = '';
            }, 5000);
        }

        // Edit email - open edit modal
        function editEmail(clientId) {
            currentEditingClientId = clientId;
            const email = emailData[clientId];
            if (!email) return;

            document.getElementById('editModalBusinessName').textContent = email.businessName;

            // Load existing extra content if available
            const extraContentInput = document.getElementById('extraContentInput');
            extraContentInput.value = extraContentData[clientId] || '';

            document.getElementById('editModal').classList.add('active');
        }

        // Save extra content
        function saveExtraContent() {
            if (!currentEditingClientId) return;

            const extraContent = document.getElementById('extraContentInput').value.trim();

            if (extraContent) {
                extraContentData[currentEditingClientId] = extraContent;
            } else {
                // Remove if empty
                delete extraContentData[currentEditingClientId];
            }

            // Save to localStorage
            localStorage.setItem(EXTRA_CONTENT_KEY, JSON.stringify(extraContentData));

            // Update UI indicator
            updateExtraContentIndicator(currentEditingClientId);

            // Show save confirmation
            document.getElementById('saveStatus').textContent = 'Extra content saved ✓';
            setTimeout(() => {
                document.getElementById('saveStatus').textContent = '';
            }, 2000);

            closeEditModal();
        }

        // Update extra content indicator (add icon to edit button)
        function updateExtraContentIndicator(clientId) {
            const row = document.getElementById('row_' + clientId);
            if (!row) return;

            const editBtn = row.querySelector('[onclick*="editEmail"]');
            if (!editBtn) return;

            if (extraContentData[clientId]) {
                // Add indicator that extra content exists
                editBtn.style.background = '#fef3c7';
                editBtn.style.borderColor = '#f59e0b';
                editBtn.title = 'Edit Email (Extra content added)';
            } else {
                // Reset to default
                editBtn.style.background = 'white';
                editBtn.style.borderColor = '#e9ecef';
                editBtn.title = 'Edit Email';
            }
        }

        // Close edit modal
        function closeEditModal() {
            document.getElementById('editModal').classList.remove('active');
            currentEditingClientId = null;
        }

        // Close edit modal on Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeEditModal();
            }
        });

        // Close edit modal on backdrop click
        document.getElementById('editModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeEditModal();
            }
        });

        // Initialize on page load
        window.addEventListener('DOMContentLoaded', function() {
            loadApprovals();
        });

"""

        # Add email data as JavaScript
        for email in all_emails:
            business_name = email.get('business_name', 'Unknown')
            recipient_email = email.get('recipient_email', '').lower()
            report_type = email.get('report_type', 'Unknown')
            # Include report type in ID to distinguish SEO vs Google Ads for same business
            safe_id = f"{business_name}_{report_type}".replace(' ', '_').replace("'", "\\'")

            # Determine agency based on recipient email
            agency = 'Direct Clients'
            if 'caroline@cju.net.au' in recipient_email:
                agency = 'CJU'
            elif 'tk@innovait.com.au' in recipient_email:
                agency = 'InnovaIT'
            elif 'natasha@lunadigitalmarketing.com.au' in recipient_email:
                agency = 'Luna'
            elif 'caroline.ucherek@acurio.com.au' in recipient_email:
                agency = 'Acurio'

            # Get PDF path and convert backslashes to forward slashes for JavaScript
            pdf_path_js = email.get('pdf_path', '').replace('\\', '/')

            html += f"""
        emailData['{safe_id}'] = {{
            businessName: '{business_name}',
            subject: `{email.get('subject', '')}`,
            htmlBody: `{self._escape_js_string(email.get('html_body', ''))}`,
            pdfPath: '{pdf_path_js}',
            reportType: '{email.get('report_type', 'Unknown')}',
            agency: '{agency}'
        }};
"""

        html += """

        // Modal functions
        function showEmail(clientId) {
            const email = emailData[clientId];
            if (!email) return;

            document.getElementById('modalBusinessName').textContent = email.businessName;
            document.getElementById('modalSubject').textContent = email.subject;
            document.getElementById('modalEmailContent').innerHTML = email.htmlBody;

            // Set PDF link with URL encoding for spaces
            const pdfLink = document.getElementById('modalPdfLink');
            let pdfPath = email.pdfPath.replace(/\\\\/g, '/');

            // Ensure uppercase drive letter
            if (pdfPath.length >= 2 && pdfPath[1] === ':') {
                pdfPath = pdfPath[0].toUpperCase() + pdfPath.substring(1);
            }

            // Encode only spaces and special chars, not slashes or colons in drive letter
            const encodedPath = pdfPath.replace(/ /g, '%20');
            pdfLink.href = 'file:///' + encodedPath;

            document.getElementById('emailModal').classList.add('active');
        }

        function closeModal() {
            document.getElementById('emailModal').classList.remove('active');
        }

        document.getElementById('emailModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });

        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });

        // Search and filter functionality
        const searchInput = document.getElementById('searchInput');
        const filterBtns = document.querySelectorAll('.filter-btn');
        const viewBtns = document.querySelectorAll('.view-btn');
        const tableView = document.getElementById('tableView');
        const gridView = document.getElementById('gridView');
        const resultCount = document.getElementById('resultCount');
        const noResults = document.getElementById('noResults');

        let currentTypeFilter = 'all';
        let currentAgencyFilter = 'all';
        let currentView = 'table';

        // Search functionality
        searchInput.addEventListener('input', function(e) {
            const searchTerm = e.target.value.toLowerCase();
            applyFilters(searchTerm);
        });

        // Filter buttons
        filterBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                // Determine if this is a type or agency filter
                const filterType = this.dataset.filterType;
                const filterAgency = this.dataset.filterAgency;

                if (filterType) {
                    // Type filter
                    document.querySelectorAll('[data-filter-type]').forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                    currentTypeFilter = filterType;
                } else if (filterAgency) {
                    // Agency filter
                    document.querySelectorAll('[data-filter-agency]').forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                    currentAgencyFilter = filterAgency;
                }

                applyFilters(searchInput.value.toLowerCase());
            });
        });

        // View toggle
        viewBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                viewBtns.forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                currentView = this.dataset.view;

                if (currentView === 'table') {
                    tableView.style.display = 'block';
                    gridView.classList.remove('active');
                } else {
                    tableView.style.display = 'none';
                    gridView.classList.add('active');
                }
            });
        });

        // Apply filters
        function applyFilters(searchTerm = '') {
            let visibleCount = 0;

            // Filter table rows
            const tableRows = document.querySelectorAll('#reportsTableBody tr');
            tableRows.forEach(row => {
                const businessName = row.dataset.business.toLowerCase();
                const contact = row.dataset.contact.toLowerCase();
                const email = row.dataset.email.toLowerCase();
                const reportType = row.dataset.type;
                const agency = row.dataset.agency;

                const matchesSearch = searchTerm === '' ||
                    businessName.includes(searchTerm) ||
                    contact.includes(searchTerm) ||
                    email.includes(searchTerm);

                const matchesTypeFilter = currentTypeFilter === 'all' || reportType === currentTypeFilter;
                const matchesAgencyFilter = currentAgencyFilter === 'all' || agency === currentAgencyFilter;

                if (matchesSearch && matchesTypeFilter && matchesAgencyFilter) {
                    row.classList.remove('hidden');
                    visibleCount++;
                } else {
                    row.classList.add('hidden');
                }
            });

            // Filter grid cards
            const gridCards = document.querySelectorAll('.client-grid .client-card');
            gridCards.forEach(card => {
                const businessName = card.dataset.business.toLowerCase();
                const contact = card.dataset.contact.toLowerCase();
                const email = card.dataset.email.toLowerCase();
                const reportType = card.dataset.type;
                const agency = card.dataset.agency;

                const matchesSearch = searchTerm === '' ||
                    businessName.includes(searchTerm) ||
                    contact.includes(searchTerm) ||
                    email.includes(searchTerm);

                const matchesTypeFilter = currentTypeFilter === 'all' || reportType === currentTypeFilter;
                const matchesAgencyFilter = currentAgencyFilter === 'all' || agency === currentAgencyFilter;

                if (matchesSearch && matchesTypeFilter && matchesAgencyFilter) {
                    card.style.display = 'block';
                } else {
                    card.style.display = 'none';
                }
            });

            // Update result count
            resultCount.textContent = visibleCount;

            // Show/hide no results message
            if (visibleCount === 0) {
                noResults.style.display = 'block';
                document.querySelector('table').style.display = 'none';
            } else {
                noResults.style.display = 'none';
                document.querySelector('table').style.display = 'table';
            }
        }

        // Table sorting
        let currentSortColumn = 'business';
        let sortAscending = true;

        document.querySelectorAll('th[data-sort]').forEach(th => {
            th.addEventListener('click', function() {
                const sortKey = this.dataset.sort;

                if (currentSortColumn === sortKey) {
                    sortAscending = !sortAscending;
                } else {
                    currentSortColumn = sortKey;
                    sortAscending = true;
                }

                // Update sort icons
                document.querySelectorAll('th').forEach(header => {
                    header.classList.remove('sorted');
                    const icon = header.querySelector('.sort-icon');
                    if (icon) icon.textContent = '▼';
                });

                this.classList.add('sorted');
                const icon = this.querySelector('.sort-icon');
                if (icon) icon.textContent = sortAscending ? '▼' : '▲';

                sortTable(sortKey, sortAscending);
            });
        });

        function sortTable(sortKey, ascending) {
            const tbody = document.getElementById('reportsTableBody');
            const rows = Array.from(tbody.querySelectorAll('tr'));

            rows.sort((a, b) => {
                let aVal = a.dataset[sortKey] || '';
                let bVal = b.dataset[sortKey] || '';

                if (aVal < bVal) return ascending ? -1 : 1;
                if (aVal > bVal) return ascending ? 1 : -1;
                return 0;
            });

            rows.forEach(row => tbody.appendChild(row));
        }

        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Don't trigger shortcuts when typing in input/textarea
            const isInputField = e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA';

            // Focus search: Ctrl/Cmd + K
            if ((e.ctrlKey || e.metaKey) && e.key === 'k' && !isInputField) {
                e.preventDefault();
                searchInput.focus();
            }
            // Note: Removed Ctrl+V shortcut - Ctrl+V should ONLY be used for paste
        });
    </script>
</body>
</html>
"""

        return html

    def _generate_table_row(self, email: Dict) -> str:
        """Generate HTML for a single table row."""
        from urllib.parse import quote

        business_name = email.get('business_name', 'Unknown')
        recipient_name = email.get('recipient_name', 'Unknown')
        recipient_email = email.get('recipient_email', '')
        report_type = email.get('report_type', 'Unknown')
        pdf_path = email.get('pdf_path', '')

        # Determine agency based on recipient email
        agency = 'Direct Clients'
        email_lower = recipient_email.lower()
        if 'caroline@cju.net.au' in email_lower:
            agency = 'CJU'
        elif 'tk@innovait.com.au' in email_lower:
            agency = 'InnovaIT'
        elif 'natasha@lunadigitalmarketing.com.au' in email_lower:
            agency = 'Luna'
        elif 'caroline.ucherek@acurio.com.au' in email_lower:
            agency = 'Acurio'

        # Create safe ID for JavaScript - include report type to distinguish SEO vs Google Ads
        safe_id = f"{business_name}_{report_type}".replace(' ', '_').replace("'", "\\'")

        # URL encode the PDF path - only encode spaces, not slashes
        pdf_url = pdf_path.replace(chr(92), '/')
        # Ensure uppercase drive letter
        if len(pdf_url) >= 2 and pdf_url[1] == ':':
            pdf_url = pdf_url[0].upper() + pdf_url[1:]
        # Only encode spaces, not slashes or colons
        pdf_url_encoded = pdf_url.replace(' ', '%20')

        # Badge class based on report type
        badge_class = 'seo' if report_type == 'SEO' else 'sem'

        return f"""
                    <tr data-business="{business_name}" data-contact="{recipient_name}" data-email="{recipient_email}" data-type="{report_type}" data-agency="{agency}" data-status="pending" id="row_{safe_id}">
                        <td class="business-name">{business_name}</td>
                        <td>{recipient_name}</td>
                        <td>{recipient_email}</td>
                        <td><span class="badge {badge_class}">{report_type}</span></td>
                        <td>
                            <span class="status-badge status-pending" id="status_{safe_id}">Pending</span>
                        </td>
                        <td>
                            <div class="approval-btns">
                                <button class="approve-btn" onclick="setStatus('{safe_id}', 'approved')" title="Approve">✓ Approve</button>
                                <button class="fix-btn" onclick="setStatus('{safe_id}', 'needs-fix')" title="Needs Fix">✗ Needs Fix</button>
                            </div>
                        </td>
                        <td>
                            <div class="action-btns">
                                <button class="icon-btn" onclick="showEmail('{safe_id}')" title="Preview Email">📧</button>
                                <button class="icon-btn" onclick="editEmail('{safe_id}')" title="Edit Email">✏️</button>
                                <a href="file:///{pdf_url_encoded}" class="icon-btn" target="_blank" title="View PDF">📄</a>
                            </div>
                        </td>
                    </tr>
"""

    def _generate_client_card(self, email: Dict) -> str:
        """Generate HTML for a single client card."""
        from urllib.parse import quote

        business_name = email.get('business_name', 'Unknown')
        recipient_name = email.get('recipient_name', 'Unknown')
        recipient_email = email.get('recipient_email', '')
        report_type = email.get('report_type', 'Unknown')
        pdf_path = email.get('pdf_path', '')

        # Determine agency based on recipient email
        agency = 'Direct Clients'
        email_lower = recipient_email.lower()
        if 'caroline@cju.net.au' in email_lower:
            agency = 'CJU'
        elif 'tk@innovait.com.au' in email_lower:
            agency = 'InnovaIT'
        elif 'natasha@lunadigitalmarketing.com.au' in email_lower:
            agency = 'Luna'
        elif 'caroline.ucherek@acurio.com.au' in email_lower:
            agency = 'Acurio'

        # Create safe ID for JavaScript - include report type to distinguish SEO vs Google Ads
        safe_id = f"{business_name}_{report_type}".replace(' ', '_').replace("'", "\\'")

        # URL encode the PDF path - only encode spaces, not slashes
        pdf_url = pdf_path.replace(chr(92), '/')
        # Ensure uppercase drive letter
        if len(pdf_url) >= 2 and pdf_url[1] == ':':
            pdf_url = pdf_url[0].upper() + pdf_url[1:]
        # Only encode spaces, not slashes or colons
        pdf_url_encoded = pdf_url.replace(' ', '%20')

        # Badge class based on report type
        badge_class = 'seo' if report_type == 'SEO' else 'sem'

        return f"""
                <div class="client-card" data-business="{business_name}" data-contact="{recipient_name}" data-email="{recipient_email}" data-type="{report_type}" data-agency="{agency}">
                    <div class="business-name">{business_name}</div>
                    <div class="contact">{recipient_name} ({recipient_email})</div>
                    <div style="margin-bottom: 12px;">
                        <span class="badge {badge_class}">{report_type}</span>
                    </div>
                    <div class="actions">
                        <button class="btn btn-primary" onclick="showEmail('{safe_id}')">
                            📧 Preview Email
                        </button>
                        <a href="file:///{pdf_url_encoded}" class="btn btn-secondary" target="_blank">
                            📄 View PDF
                        </a>
                    </div>
                </div>
"""

    def _escape_js_string(self, text: str) -> str:
        """Escape string for safe inclusion in JavaScript."""
        # Replace backticks and special characters
        text = text.replace('`', '\\`')
        text = text.replace('${', '\\${')
        text = text.replace('\\', '\\\\')
        return text
