"""
Test script to verify Google Drive month-based folder setup.
"""

import os
import sys
from datetime import datetime
from dateutil.relativedelta import relativedelta
from dotenv import load_dotenv
from src.drive_reader import DriveReader

# Fix Windows console encoding for Unicode characters
if sys.platform == 'win32':
    import codecs
    sys.stdout = codecs.getwriter('utf-8')(sys.stdout.buffer, 'strict')
    sys.stderr = codecs.getwriter('utf-8')(sys.stderr.buffer, 'strict')

def test_month_detection():
    """Test automatic month detection."""
    last_month = datetime.now() - relativedelta(months=1)
    month = last_month.strftime('%B')

    print("\n" + "="*60)
    print("MONTH DETECTION TEST")
    print("="*60)
    print(f"Current date: {datetime.now().strftime('%B %d, %Y')}")
    print(f"Detected previous month: {month}")
    print("="*60 + "\n")

def test_drive_connection():
    """Test Google Drive connection and folder access."""
    load_dotenv()

    print("\n" + "="*60)
    print("GOOGLE DRIVE CONNECTION TEST")
    print("="*60)

    # Load folder IDs from environment
    seo_folder = os.getenv('DRIVE_SEO_FOLDER_ID')
    sem_folder = os.getenv('DRIVE_SEM_FOLDER_ID')

    print(f"\nSEO Folder ID: {seo_folder}")
    print(f"SEM Folder ID: {sem_folder}")

    if not seo_folder or not sem_folder:
        print("\n❌ ERROR: Folder IDs not found in .env")
        print("Please add DRIVE_SEO_FOLDER_ID and DRIVE_SEM_FOLDER_ID to your .env file")
        return False

    try:
        # Initialize Drive reader
        print("\nInitializing Google Drive connection...")
        drive_reader = DriveReader()
        print("✓ Connected to Google Drive")

        # Test month detection
        last_month = datetime.now() - relativedelta(months=1)
        month = last_month.strftime('%B')

        # Test SEO folder
        print(f"\nLooking for '{month}' folder in SEO Reports...")
        seo_month_folder = drive_reader.find_subfolder_by_name(seo_folder, month)

        if seo_month_folder:
            print(f"✓ Found SEO '{month}' folder: {seo_month_folder}")

            # List PDFs in folder
            pdfs = drive_reader.list_pdfs_in_folder(seo_month_folder)
            print(f"  → Contains {len(pdfs)} PDF files")
        else:
            print(f"⚠ SEO '{month}' folder not found")
            print(f"  Please create: Shared drives\\DWS\\SEO\\Reports\\2025\\{month}\\")

        # Test SEM folder
        print(f"\nLooking for '{month}' folder in Google Ads Reports...")
        sem_month_folder = drive_reader.find_subfolder_by_name(sem_folder, month)

        if sem_month_folder:
            print(f"✓ Found Google Ads '{month}' folder: {sem_month_folder}")

            # List PDFs in folder
            pdfs = drive_reader.list_pdfs_in_folder(sem_month_folder)
            print(f"  → Contains {len(pdfs)} PDF files")
        else:
            print(f"⚠ Google Ads '{month}' folder not found")
            print(f"  Please create: Shared drives\\DWS\\Google Ads\\Reports\\2025\\{month}\\")

        print("\n" + "="*60)
        print("TEST COMPLETE")
        print("="*60 + "\n")

        return True

    except Exception as e:
        print(f"\n❌ ERROR: {e}")
        print("\nPossible issues:")
        print("1. token.json missing or invalid (run main.py first)")
        print("2. Google Drive API not enabled in Google Cloud Console")
        print("3. Incorrect folder IDs in .env file")
        return False

if __name__ == '__main__':
    print("\n")
    print("+" + "="*58 + "+")
    print("|" + " "*15 + "GOOGLE DRIVE SETUP TEST" + " "*20 + "|")
    print("+" + "="*58 + "+")

    # Test month detection
    test_month_detection()

    # Test Drive connection
    success = test_drive_connection()

    if success:
        print("✓ All tests passed! Your Google Drive setup is ready.")
        print("\nNext steps:")
        print("1. Ensure month folders contain PDF reports")
        print("2. Run: python main.py --full")
    else:
        print("✗ Setup incomplete. Please resolve the issues above.")
