"""
Integration Tests for Email Reports Automation System.
Tests end-to-end workflow with mock data.
"""

import unittest
from unittest.mock import Mock, MagicMock, patch, mock_open
import sys
from pathlib import Path
import io
import tempfile
import shutil

# Add src to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from src.pdf_extractor import PDFExtractor
from src.client_database import ClientDatabase
from src.email_generator import EmailGenerator
from src.gmail_reader import GmailReader
from src.gmail_sender import GmailSender
from src.approval_tracker import ApprovalTracker


class TestIntegrationHappyPath(unittest.TestCase):
    """Integration test for happy path - all systems working correctly."""

    def setUp(self):
        """Set up test fixtures."""
        self.temp_dir = tempfile.mkdtemp()
        self.sample_csv_data = """Client-Name,Contact-Name,Contact-Email,Service-Type,SEO-Introduction,Google-Ads-Introduction
The George Centre,John Smith,john@example.com,SEO,Great work on content updates.,
ABC Corporation,Jane Doe,jane@abc.com,SEM,,Your ad copy is performing well.
Tech Solutions LLC,Mike Johnson,mike@tech.com,SEO,Technical improvements paying off.,
"""

    def tearDown(self):
        """Clean up test fixtures."""
        shutil.rmtree(self.temp_dir, ignore_errors=True)

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    @patch('pdfplumber.open')
    def test_end_to_end_workflow_success(self, mock_pdf_open, mock_file, mock_exists):
        """Test complete workflow from PDF extraction to email generation."""
        # Setup mocks
        mock_exists.return_value = True
        mock_file.return_value = io.StringIO(self.sample_csv_data)

        # Mock PDF extraction
        mock_page = MagicMock()
        mock_page.extract_text.return_value = """
        The George Centre
        2024 October
        Sessions 500
        Active users 450
        """
        mock_page.extract_tables.return_value = [
            [['Metric', 'Value', 'Change'],
             ['Sessions', '500', '+10%'],
             ['Active users', '450', '+5%'],
             ['New users', '300', '+8%'],
             ['Key events', '25', '+12%'],
             ['Engagement rate', '55.5%', '+3%'],
             ['Bounce rate', '35.2%', '-2%'],
             ['Average session duration', '00:02:30', '+5%']]
        ]

        mock_pdf = MagicMock()
        mock_pdf.pages = [mock_page]
        mock_pdf.__enter__ = Mock(return_value=mock_pdf)
        mock_pdf.__exit__ = Mock(return_value=False)
        mock_pdf_open.return_value = mock_pdf

        # Step 1: Extract PDF data
        extractor = PDFExtractor()
        pdf_data = extractor.extract_report_data('test_seo.pdf')

        self.assertEqual(pdf_data['report_type'], 'SEO')
        self.assertIsNotNone(pdf_data['business_name'])
        self.assertEqual(len(pdf_data['extraction_errors']), 0)

        # Step 2: Match to client database
        db = ClientDatabase('test_clients.csv')
        client = db.find_client(pdf_data['business_name'])

        self.assertIsNotNone(client)
        self.assertEqual(client.get('Contact-Name'), 'John Smith')

        # Step 3: Generate email
        generator = EmailGenerator('templates/email_template.html')
        # We'll skip actual email generation in this mock test
        # but verify that we have all required data
        self.assertIsNotNone(pdf_data['kpis'])
        self.assertIsNotNone(client)


class TestIntegrationPartialFailure(unittest.TestCase):
    """Integration test for partial failure scenarios."""

    def setUp(self):
        """Set up test fixtures."""
        self.sample_csv_data = """Client-Name,Contact-Name,Contact-Email,Service-Type,SEO-Introduction,Google-Ads-Introduction
The George Centre,John Smith,john@example.com,SEO,Great work.,
ABC Corporation,Jane Doe,jane@abc.com,SEM,,Ads working well.
"""

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    @patch('pdfplumber.open')
    def test_pdf_parsing_failure_recovery(self, mock_pdf_open, mock_file, mock_exists):
        """Test system handles PDF parsing failure gracefully."""
        mock_exists.return_value = True
        mock_file.return_value = io.StringIO(self.sample_csv_data)

        # Mock corrupted PDF
        mock_pdf_open.side_effect = Exception("Corrupted PDF file")

        extractor = PDFExtractor()
        pdf_data = extractor.extract_report_data('corrupted.pdf')

        # Should return result with errors, not crash
        self.assertIsNotNone(pdf_data)
        self.assertIn('extraction_errors', pdf_data)
        self.assertGreater(len(pdf_data['extraction_errors']), 0)

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    @patch('pdfplumber.open')
    def test_client_not_found_in_database(self, mock_pdf_open, mock_file, mock_exists):
        """Test handling when PDF business name doesn't match any client."""
        mock_exists.return_value = True
        mock_file.return_value = io.StringIO(self.sample_csv_data)

        # Mock PDF with unknown business name
        mock_page = MagicMock()
        mock_page.extract_text.return_value = """
        Unknown Business LLC
        2024 October
        Sessions 500
        """
        mock_page.extract_tables.return_value = []

        mock_pdf = MagicMock()
        mock_pdf.pages = [mock_page]
        mock_pdf.__enter__ = Mock(return_value=mock_pdf)
        mock_pdf.__exit__ = Mock(return_value=False)
        mock_pdf_open.return_value = mock_pdf

        # Extract PDF data
        extractor = PDFExtractor()
        pdf_data = extractor.extract_report_data('unknown_business.pdf')

        # Try to find client
        db = ClientDatabase('test_clients.csv')
        client = db.find_client(pdf_data['business_name'])

        # Should return None, not crash
        self.assertIsNone(client)


class TestIntegrationMultiplePDFs(unittest.TestCase):
    """Integration test for processing multiple PDFs."""

    def setUp(self):
        """Set up test fixtures."""
        self.sample_csv_data = """Client-Name,Contact-Name,Contact-Email,Service-Type,SEO-Introduction,Google-Ads-Introduction
Business A,Contact A,a@example.com,SEO,SEO intro A,
Business B,Contact B,b@example.com,SEM,,SEM intro B
Business C,Contact C,c@example.com,SEO,SEO intro C,
"""

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    @patch('pdfplumber.open')
    def test_process_multiple_pdfs_batch(self, mock_pdf_open, mock_file, mock_exists):
        """Test processing multiple PDFs in batch."""
        mock_exists.return_value = True
        mock_file.return_value = io.StringIO(self.sample_csv_data)

        # Mock three different PDFs
        def create_mock_pdf(business_name, report_type):
            mock_page = MagicMock()
            mock_page.extract_text.return_value = f"""
            {business_name}
            2024 October
            Sessions 500
            """
            mock_page.extract_tables.return_value = []

            mock_pdf = MagicMock()
            mock_pdf.pages = [mock_page]
            return mock_pdf

        pdfs = [
            ('business_a.pdf', 'Business A', 'SEO'),
            ('business_b.pdf', 'Business B', 'Google Ads'),
            ('business_c.pdf', 'Business C', 'SEO')
        ]

        extractor = PDFExtractor()
        db = ClientDatabase('test_clients.csv')

        results = []
        for pdf_file, business_name, report_type in pdfs:
            # Mock PDF for this iteration
            mock_pdf = create_mock_pdf(business_name, report_type)
            mock_pdf.__enter__ = Mock(return_value=mock_pdf)
            mock_pdf.__exit__ = Mock(return_value=False)
            mock_pdf_open.return_value = mock_pdf

            # Extract and match
            pdf_data = extractor.extract_report_data(pdf_file)
            client = db.find_client(pdf_data['business_name'])

            results.append({
                'pdf_file': pdf_file,
                'pdf_data': pdf_data,
                'client': client,
                'success': client is not None
            })

        # All three should be processed
        self.assertEqual(len(results), 3)

        # All should have found matching clients
        successful = [r for r in results if r['success']]
        self.assertEqual(len(successful), 3)


class TestIntegrationErrorScenarios(unittest.TestCase):
    """Integration tests for various error scenarios."""

    @patch('pathlib.Path.exists')
    def test_client_database_file_not_found(self, mock_exists):
        """Test handling when client database file doesn't exist."""
        mock_exists.return_value = False

        with self.assertRaises(FileNotFoundError):
            db = ClientDatabase('nonexistent.csv')

    @patch('pdfplumber.open')
    def test_pdf_file_not_found(self, mock_pdf_open):
        """Test handling when PDF file doesn't exist."""
        mock_pdf_open.side_effect = FileNotFoundError("PDF not found")

        extractor = PDFExtractor()
        result = extractor.extract_report_data('nonexistent.pdf')

        self.assertIn('extraction_errors', result)
        self.assertGreater(len(result['extraction_errors']), 0)

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    def test_malformed_csv_data(self, mock_file, mock_exists):
        """Test handling of malformed CSV data."""
        malformed_csv = """This is not,a valid,CSV structure
Missing headers
And random data
"""
        mock_exists.return_value = True
        mock_file.return_value = io.StringIO(malformed_csv)

        # Should either raise exception or handle gracefully
        try:
            db = ClientDatabase('malformed.csv')
            # If it loads, should have minimal or no clients
            self.assertIsInstance(db.clients, list)
        except Exception as e:
            # If it raises exception, that's also acceptable
            self.assertIsNotNone(e)


class TestIntegrationDataConsistency(unittest.TestCase):
    """Integration tests for data consistency across workflow."""

    def setUp(self):
        """Set up test fixtures."""
        self.sample_csv_data = """Client-Name,Contact-Name,Contact-Email,Service-Type,SEO-Introduction,Google-Ads-Introduction
The George Centre,John Smith,john@example.com,SEO,Great work on content.,
"""

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    @patch('pdfplumber.open')
    def test_service_type_consistency(self, mock_pdf_open, mock_file, mock_exists):
        """Test that service type is consistent from PDF to client match."""
        mock_exists.return_value = True
        mock_file.return_value = io.StringIO(self.sample_csv_data)

        # Mock SEO PDF
        mock_page = MagicMock()
        mock_page.extract_text.return_value = """
        The George Centre
        2024 October
        Sessions 500
        """
        mock_page.extract_tables.return_value = []

        mock_pdf = MagicMock()
        mock_pdf.pages = [mock_page]
        mock_pdf.__enter__ = Mock(return_value=mock_pdf)
        mock_pdf.__exit__ = Mock(return_value=False)
        mock_pdf_open.return_value = mock_pdf

        # Extract data
        extractor = PDFExtractor()
        pdf_data = extractor.extract_report_data('test.pdf')

        # Find client
        db = ClientDatabase('test_clients.csv')
        client = db.find_client(pdf_data['business_name'])

        # Verify service types align
        pdf_type = pdf_data['report_type']
        client_type = client.get('Service-Type') if client else None

        # SEO report should match SEO client
        if pdf_type == 'SEO':
            self.assertEqual(client_type, 'SEO')

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    @patch('pdfplumber.open')
    def test_date_format_consistency(self, mock_pdf_open, mock_file, mock_exists):
        """Test that date format is extracted consistently."""
        mock_exists.return_value = True
        mock_file.return_value = io.StringIO(self.sample_csv_data)

        # Mock PDF with date
        mock_page = MagicMock()
        mock_page.extract_text.return_value = """
        The George Centre
        2024 October
        Sessions 500
        """
        mock_page.extract_tables.return_value = []

        mock_pdf = MagicMock()
        mock_pdf.pages = [mock_page]
        mock_pdf.__enter__ = Mock(return_value=mock_pdf)
        mock_pdf.__exit__ = Mock(return_value=False)
        mock_pdf_open.return_value = mock_pdf

        # Extract data
        extractor = PDFExtractor()
        pdf_data = extractor.extract_report_data('test.pdf')

        # Verify date and month are extracted
        self.assertIsNotNone(pdf_data.get('report_date'))
        self.assertIsNotNone(pdf_data.get('report_month'))


class TestIntegrationLogging(unittest.TestCase):
    """Integration tests for logging throughout workflow."""

    @patch('pathlib.Path.exists')
    @patch('builtins.open', new_callable=mock_open)
    @patch('pdfplumber.open')
    def test_logging_during_workflow(self, mock_pdf_open, mock_file, mock_exists):
        """Test that logging occurs at key workflow points."""
        mock_exists.return_value = True
        csv_data = """Client-Name,Contact-Name,Contact-Email,Service-Type,SEO-Introduction,Google-Ads-Introduction
Test Client,Test Contact,test@example.com,SEO,Test intro,
"""
        mock_file.return_value = io.StringIO(csv_data)

        # Mock PDF
        mock_page = MagicMock()
        mock_page.extract_text.return_value = "Test Client\n2024 October\nSessions 500"
        mock_page.extract_tables.return_value = []

        mock_pdf = MagicMock()
        mock_pdf.pages = [mock_page]
        mock_pdf.__enter__ = Mock(return_value=mock_pdf)
        mock_pdf.__exit__ = Mock(return_value=False)
        mock_pdf_open.return_value = mock_pdf

        # Execute workflow steps (logging should happen internally)
        extractor = PDFExtractor()
        pdf_data = extractor.extract_report_data('test.pdf')

        db = ClientDatabase('test_clients.csv')
        client = db.find_client(pdf_data['business_name'])

        # If we get here without exceptions, logging worked
        self.assertTrue(True)


if __name__ == '__main__':
    unittest.main()
