#!/usr/bin/env python3
"""
Cross-Platform Compatibility Checker
Scans all Python source files for potential Windows/Linux compatibility issues.
"""

import os
import re
from pathlib import Path
from typing import List, Dict, Tuple

class CrossPlatformChecker:
    def __init__(self, project_root: str):
        self.project_root = Path(project_root)
        self.issues = []

    def check_all_files(self) -> Dict[str, List[Dict]]:
        """Check all Python files for compatibility issues."""
        results = {}

        # Find all Python files
        for py_file in self.project_root.rglob("*.py"):
            if "__pycache__" in str(py_file) or "venv" in str(py_file):
                continue

            issues = self.check_file(py_file)
            if issues:
                results[str(py_file.relative_to(self.project_root))] = issues

        return results

    def check_file(self, filepath: Path) -> List[Dict]:
        """Check a single file for compatibility issues."""
        issues = []

        try:
            with open(filepath, 'r', encoding='utf-8') as f:
                content = f.read()
                lines = content.split('\n')

            # Check for hardcoded Windows paths
            issues.extend(self._check_hardcoded_paths(lines, filepath))

            # Check for incorrect path separators
            issues.extend(self._check_path_separators(lines, filepath))

            # Check for Windows-specific modules
            issues.extend(self._check_windows_modules(lines, filepath))

            # Check for line ending issues
            issues.extend(self._check_line_endings(filepath))

            # Check for proper path handling
            issues.extend(self._check_path_handling(lines, filepath))

        except Exception as e:
            issues.append({
                'type': 'ERROR',
                'line': 0,
                'message': f"Error reading file: {e}"
            })

        return issues

    def _check_hardcoded_paths(self, lines: List[str], filepath: Path) -> List[Dict]:
        """Check for hardcoded Windows or Linux paths."""
        issues = []

        # Patterns for hardcoded paths
        windows_path_pattern = r'["\'][A-Z]:\\[^"\']*["\']'
        linux_path_pattern = r'["\']\/home\/[^"\']*["\']'

        for i, line in enumerate(lines, 1):
            # Skip comments
            if line.strip().startswith('#'):
                continue

            # Check for Windows paths
            if re.search(windows_path_pattern, line):
                issues.append({
                    'type': 'WARNING',
                    'line': i,
                    'message': f"Hardcoded Windows path detected: {line.strip()}"
                })

            # Check for Linux paths
            if re.search(linux_path_pattern, line):
                issues.append({
                    'type': 'INFO',
                    'line': i,
                    'message': f"Hardcoded Linux path detected: {line.strip()}"
                })

        return issues

    def _check_path_separators(self, lines: List[str], filepath: Path) -> List[Dict]:
        """Check for incorrect use of path separators."""
        issues = []

        for i, line in enumerate(lines, 1):
            # Skip comments and strings that are obviously not paths
            if line.strip().startswith('#'):
                continue

            # Check for string concatenation with backslashes
            if re.search(r'[\'"]/[^/]+\\[^/]+[\'"]', line):
                issues.append({
                    'type': 'WARNING',
                    'line': i,
                    'message': f"Mixed path separators detected: {line.strip()}"
                })

            # Check for os.path.join with hardcoded separators
            if 'os.path.join' in line and ('\\\\' in line or '//' in line):
                issues.append({
                    'type': 'WARNING',
                    'line': i,
                    'message': f"os.path.join with hardcoded separator: {line.strip()}"
                })

        return issues

    def _check_windows_modules(self, lines: List[str], filepath: Path) -> List[Dict]:
        """Check for Windows-specific module imports."""
        issues = []
        windows_modules = ['winreg', 'msvcrt', 'winsound', '_winapi']

        for i, line in enumerate(lines, 1):
            for module in windows_modules:
                if f'import {module}' in line or f'from {module}' in line:
                    issues.append({
                        'type': 'ERROR',
                        'line': i,
                        'message': f"Windows-specific module '{module}' will fail on Linux"
                    })

        return issues

    def _check_line_endings(self, filepath: Path) -> List[Dict]:
        """Check for Windows line endings (CRLF)."""
        issues = []

        try:
            with open(filepath, 'rb') as f:
                content = f.read()

            if b'\r\n' in content:
                issues.append({
                    'type': 'INFO',
                    'line': 0,
                    'message': "File has Windows line endings (CRLF). May need dos2unix on Linux."
                })

        except Exception as e:
            pass

        return issues

    def _check_path_handling(self, lines: List[str], filepath: Path) -> List[Dict]:
        """Check for proper cross-platform path handling."""
        issues = []
        has_pathlib = False
        has_os_path = False

        for i, line in enumerate(lines, 1):
            # Check if file imports pathlib
            if 'from pathlib import' in line or 'import pathlib' in line:
                has_pathlib = True

            # Check if file imports os.path
            if 'import os' in line:
                has_os_path = True

            # Look for path operations without proper handling
            if any(op in line for op in ['open(', 'with open']):
                # Check if using string concatenation for paths
                if '+' in line and any(sep in line for sep in ['/', '\\\\']):
                    issues.append({
                        'type': 'WARNING',
                        'line': i,
                        'message': f"Path concatenation with + operator: {line.strip()}"
                    })

        # If file does path operations, recommend pathlib
        if not has_pathlib and not has_os_path:
            for i, line in enumerate(lines, 1):
                if any(kw in line for kw in ['open(', 'with open', 'file', 'dir']):
                    issues.append({
                        'type': 'INFO',
                        'line': 0,
                        'message': "Consider importing pathlib.Path for cross-platform paths"
                    })
                    break

        return issues

    def print_report(self, results: Dict[str, List[Dict]]):
        """Print a formatted report of compatibility issues."""
        if not results:
            print("✅ No cross-platform compatibility issues found!")
            return

        print("\n" + "="*80)
        print("CROSS-PLATFORM COMPATIBILITY REPORT")
        print("="*80)

        error_count = 0
        warning_count = 0
        info_count = 0

        for file, issues in sorted(results.items()):
            print(f"\n📄 {file}")
            print("-" * 80)

            for issue in issues:
                icon = {
                    'ERROR': '❌',
                    'WARNING': '⚠️',
                    'INFO': 'ℹ️'
                }.get(issue['type'], '•')

                if issue['line'] > 0:
                    print(f"  {icon} Line {issue['line']}: {issue['message']}")
                else:
                    print(f"  {icon} {issue['message']}")

                # Count issues by severity
                if issue['type'] == 'ERROR':
                    error_count += 1
                elif issue['type'] == 'WARNING':
                    warning_count += 1
                else:
                    info_count += 1

        print("\n" + "="*80)
        print("SUMMARY")
        print("="*80)
        print(f"Files checked: {len(results)}")
        print(f"❌ Errors: {error_count} (must fix before Linux deployment)")
        print(f"⚠️  Warnings: {warning_count} (should review)")
        print(f"ℹ️  Info: {info_count} (informational)")
        print("="*80 + "\n")

        if error_count > 0:
            print("❌ ERRORS FOUND: Must be fixed before deploying to Linux server!")
        elif warning_count > 0:
            print("⚠️  WARNINGS FOUND: Review before deploying to Linux server.")
        else:
            print("✅ No critical issues! Code should work on Linux.")


def main():
    project_root = Path(__file__).parent
    checker = CrossPlatformChecker(project_root)

    print("Checking cross-platform compatibility...")
    print(f"Project root: {project_root}\n")

    results = checker.check_all_files()
    checker.print_report(results)

    # Generate recommendations
    print("\nRECOMMENDATIONS:")
    print("-" * 80)
    print("1. Use pathlib.Path for all file paths")
    print("2. Use os.path.join() or Path() / operator for path construction")
    print("3. Avoid hardcoded paths - use environment variables or config files")
    print("4. Test on Linux before production deployment")
    print("5. Run dos2unix on all .py files after uploading to Linux")
    print("-" * 80)


if __name__ == "__main__":
    main()
